"""symeig benchmark functions."""

import numpy
from symeig import symeig
from testing_tools import symrand


## utils

## benchmark functions

def symeig_benchmark(dim):
    """Standard eigenvalue problem: symeig .
    First argument: matrix dimensionality"""
    a = MTX_DICT[dim][0].copy()
    out_d, out_v = symeig(a,overwrite=1)

def eig_benchmark(dim):
    """Standard eigenvalue problem: numpy.linalg.eig .
    First argument: matrix dimensionality"""
    a = MTX_DICT[dim][0].copy()
    out_d, out_v = numpy.linalg.eig(a)
 
def symeig_general_benchmark(dim):
    """General eigenvalue problem: symeig .
    First argument: matrix dimensionality"""
    a = MTX_DICT[dim][0].copy()
    b = MTX_DICT[dim][1].copy()
    out_d, out_v = symeig(a,b,overwrite=1)

def eig_general_benchmark(dim):
    """General eigenvalue problem: numpy.linalg.eig .
    First argument: matrix dimensionality"""
    try:
        import scipy.linalg
        a = MTX_DICT[dim][0].copy()
        b = MTX_DICT[dim][1].copy()
        out_d, out_v = scipy.linalg.eig(a,b)
    except ImportError:
        pass
    
def _get_random_mtx(dim):
    d = numpy.random.random(dim)
    mtx = numpy.random.random((dim, dim))
    mtx += mtx.T
    e, v = symeig(mtx, overwrite=1)
    if numpy.linalg.det(v)<0: v[:,0] = -v[:,0]
    h = numpy.dot(numpy.dot(v.T, numpy.diag(d)), v)
    # to avoid roundoff errors, symmetrize the matrix (again)
    return 0.5*(h.T+h)

symrand = _get_random_mtx
EIG_MTX_DIMS = [[2**i] for i in range(4,10)]
MTX_DICT = {}
for dim in EIG_MTX_DIMS:
    print dim
    print 'a'
    a = symrand(dim[0])
    print 'b'
    b = symrand(dim[0])+numpy.diag([2.]*dim[0])
    MTX_DICT[dim[0]] = (a, b)

# list of (benchmark function, list of arguments)
BENCH_FUNCS = [(symeig_benchmark, EIG_MTX_DIMS),
               (eig_benchmark, EIG_MTX_DIMS),
               (symeig_general_benchmark, EIG_MTX_DIMS),
               (eig_general_benchmark, EIG_MTX_DIMS)]

# function used to measure time
import time
TIMEFUNC = time.time

def timeit(func,*args,**kwargs):
    """Return function execution time in 1/100ths of a second."""
    tstart = TIMEFUNC()
    func(*args,**kwargs)
    return (TIMEFUNC()-tstart)*100.

def _random_seed():
    import sys
    seed = int(numx_rand.randint(2**31-1))
    numx_rand.seed(seed)
    sys.stderr.write("Random Seed: " + str(seed)+'\n')

def run_benchmarks(bench_funcs, time_digits=15):

    results_str = '| %%s | %%%d.2f |' % time_digits
    label_str = '| %%s | %s |' % 'Time (sec/100)'.center(time_digits)

    tstart = TIMEFUNC()

    # loop over all benchmarks functions
    for func, args_list in bench_funcs:
        # number of combinations of arguments(cases)
        ncases = len(args_list)
        funcname = func.__name__[:-10]

        # loop over all cases
        for i in range(ncases):
            args = args_list[i]

            # format description string
            descr = funcname + str(tuple(args))
            if i==0:
                # print summary table header
                descrlen = len(descr)+6
                results_strlen = time_digits+descrlen+7
                print '\nTiming results (%s, %d cases):' % (funcname, ncases)
                print func.__doc__
                print '+'+'-'*(results_strlen-2)+'+'
                print label_str % 'Description'.center(descrlen)
                print '+'+'-'*(results_strlen-2)+'+'        

            # execute function
            t = timeit(func, *args)

            # print summary table entry
            print results_str % (descr.center(descrlen), t)

        # print summary table tail
        print '+'+'-'*(results_strlen-2)+'+'
        
    print '\nTotal running time:', (TIMEFUNC()-tstart)*100.


def get_benchmarks():
    return BENCH_FUNCS

if __name__ == '__main__':
    print "Running benchmarks: "
    run_benchmarks(get_benchmarks())
