/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "common/translation.h"
#include "engines/advancedDetector.h"
#include "base/plugins.h"

#include "kyra/detection.h"
#include "kyra/detection_tables.h"
#include "kyra/kyra_v1.h"

namespace {

static const DebugChannelDef debugFlagList[] = {
	{Kyra::kDebugLevelScriptFuncs, "ScriptFuncs", "Script function debug level"},
	{Kyra::kDebugLevelScript, "Script", "Script interpreter debug level"},
	{Kyra::kDebugLevelSprites, "Sprites", "Sprite debug level"},
	{Kyra::kDebugLevelScreen, "Screen", "Screen debug level"},
	{Kyra::kDebugLevelSound, "Sound", "Sound debug level"},
	{Kyra::kDebugLevelAnimator, "Animator", "Animator debug level"},
	{Kyra::kDebugLevelMain, "Main", "Generic debug level"},
	{Kyra::kDebugLevelGUI, "GUI", "GUI debug level"},
	{Kyra::kDebugLevelSequence, "Sequence", "Sequence debug level"},
	{Kyra::kDebugLevelMovie, "Movie", "Movie debug level"},
	{Kyra::kDebugLevelTimer, "Timer", "Timer debug level"},
	DEBUG_CHANNEL_END
};

const char *const directoryGlobs[] = {
	"malcolm",
	"data", // LOL GOG release
	nullptr
};

const ADExtraGuiOptionsMap gameGuiOptions[] = {
	// Kyrandia 3 options

	{
		GAMEOPTION_KYRA3_AUDIENCE,
		{
			// I18N: Studio audience adds an applause and cheering sounds whenever
			// Malcolm makes a joke.
			_s("Studio audience"),
			_s("Enable studio audience"),
			"studio_audience",
			true,
			0,
			0
		}
	},

	{
		GAMEOPTION_KYRA3_SKIP,
		{
			// I18N: This option allows the user to skip text and cutscenes.
			_s("Skip support"),
			_s("Allow text and cutscenes to be skipped"),
			"skip_support",
			true,
			0,
			0
		}
	},

	{
		GAMEOPTION_KYRA3_HELIUM,
		{
			// I18N: Helium mode makes people sound like they've inhaled Helium.
			_s("Helium mode"),
			_s("Enable helium mode"),
			"helium_mode",
			false,
			0,
			0
		}
	},

	// LoL options

	{
		GAMEOPTION_LOL_SCROLLING,
		{
			// I18N: When enabled, this option makes scrolling smoother when
			// changing from one screen to another.
			_s("Smooth scrolling"),
			_s("Enable smooth scrolling when walking"),
			"smooth_scrolling",
			true,
			0,
			0
		}
	},

	{
		GAMEOPTION_LOL_CURSORS,
		{
			// I18N: When enabled, this option changes the cursor when it floats to the
			// edge of the screen to a directional arrow. The player can then click to
			// walk towards that direction.
			_s("Floating cursors"),
			_s("Enable floating cursors"),
			"floating_cursors",
			false,
			0,
			0
		}
	},

	{
		GAMEOPTION_LOL_SAVENAMES,
		{
			// I18N: When enabled, this option will fill in an autogenerated savegame
			// description into the input prompt where.
			_s("Suggest save names"),
			_s("Autogenerated naming suggestions for savegames"),
			"auto_savenames",
			false,
			0,
			0
		}
	},

	// EoB options

	{
		GAMEOPTION_EOB_HPGRAPHS,
		{
			// I18N: HP stands for Hit Points
			_s("HP bar graphs"),
			_s("Enable hit point bar graphs"),
			"hpbargraphs",
			true,
			0,
			0
		}
	},

	{
		GAMEOPTION_EOB_MOUSESWAP,
		{
			// I18N: L/R stands for Left/Right
			_s("Fight Button L/R Swap"),
			_s("Left button to attack, right button to pick up items"),
			"mousebtswap",
			false,
			0,
			0
		}
	},

	AD_EXTRA_GUI_OPTIONS_TERMINATOR
};

} // End of anonymous namespace

class KyraMetaEngineDetection : public AdvancedMetaEngineDetection {
public:
	KyraMetaEngineDetection() : AdvancedMetaEngineDetection(adGameDescs, sizeof(KYRAGameDescription), gameList, gameGuiOptions) {
		_md5Bytes = 1024 * 1024;
		_maxScanDepth = 2;
		_directoryGlobs = directoryGlobs;
	}

	const char *getEngineId() const override {
		return "kyra";
	}

	const char *getName() const override {
		return "Kyra";
	}

	const DebugChannelDef *getDebugChannels() const override {
		return debugFlagList;
	}

	const char *getOriginalCopyright() const override {
		return "The Legend of Kyrandia (C) Westwood Studios"
#ifdef ENABLE_LOL
		       "\nLands of Lore (C) Westwood Studios"
#endif
#ifdef ENABLE_EOB
		       "\nEye of the Beholder (C) TSR, Inc., (C) Strategic Simulations, Inc."
#endif
		       ;
	}
};

REGISTER_PLUGIN_STATIC(KYRA_DETECTION, PLUGIN_TYPE_ENGINE_DETECTION, KyraMetaEngineDetection);
