# coding: utf-8

# (C) Copyright IBM Corp. 2019, 2022.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# IBM OpenAPI SDK Code Generator Version: 3.46.0-a4e29da0-20220224-210428
"""
The IBM Watson&trade; Assistant service combines machine learning, natural language
understanding, and an integrated dialog editor to create conversation flows between your
apps and your users.
The Assistant v2 API provides runtime methods your client application can use to send user
input to an assistant and receive a response.

API Version: 2.0
See: https://cloud.ibm.com/docs/assistant
"""

from enum import Enum
from typing import Dict, List
import json
import sys

from ibm_cloud_sdk_core import BaseService, DetailedResponse
from ibm_cloud_sdk_core.authenticators.authenticator import Authenticator
from ibm_cloud_sdk_core.get_authenticator import get_authenticator_from_environment
from ibm_cloud_sdk_core.utils import convert_model

from .common import get_sdk_headers

##############################################################################
# Service
##############################################################################


class AssistantV2(BaseService):
    """The Assistant V2 service."""

    DEFAULT_SERVICE_URL = 'https://api.us-south.assistant.watson.cloud.ibm.com'
    DEFAULT_SERVICE_NAME = 'assistant'

    def __init__(
        self,
        version: str,
        authenticator: Authenticator = None,
        service_name: str = DEFAULT_SERVICE_NAME,
    ) -> None:
        """
        Construct a new client for the Assistant service.

        :param str version: Release date of the API version you want to use.
               Specify dates in YYYY-MM-DD format. The current version is `2021-11-27`.

        :param Authenticator authenticator: The authenticator specifies the authentication mechanism.
               Get up to date information from https://github.com/IBM/python-sdk-core/blob/main/README.md
               about initializing the authenticator of your choice.
        """
        if version is None:
            raise ValueError('version must be provided')

        if not authenticator:
            authenticator = get_authenticator_from_environment(service_name)
        BaseService.__init__(self,
                             service_url=self.DEFAULT_SERVICE_URL,
                             authenticator=authenticator)
        self.version = version
        self.configure_service(service_name)

    #########################
    # Sessions
    #########################

    def create_session(self, assistant_id: str, **kwargs) -> DetailedResponse:
        """
        Create a session.

        Create a new session. A session is used to send user input to a skill and receive
        responses. It also maintains the state of the conversation. A session persists
        until it is deleted, or until it times out because of inactivity. (For more
        information, see the
        [documentation](https://cloud.ibm.com/docs/assistant?topic=assistant-assistant-settings).

        :param str assistant_id: Unique identifier of the assistant. To find the
               assistant ID in the Watson Assistant user interface, open the assistant
               settings and click **API Details**. For information about creating
               assistants, see the
               [documentation](https://cloud.ibm.com/docs/assistant?topic=assistant-assistant-add#assistant-add-task).
               **Note:** Currently, the v2 API does not support creating assistants.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SessionResponse` object
        """

        if assistant_id is None:
            raise ValueError('assistant_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V2',
                                      operation_id='create_session')
        headers.update(sdk_headers)

        params = {'version': self.version}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        path_param_keys = ['assistant_id']
        path_param_values = self.encode_path_vars(assistant_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/v2/assistants/{assistant_id}/sessions'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response

    def delete_session(self, assistant_id: str, session_id: str,
                       **kwargs) -> DetailedResponse:
        """
        Delete session.

        Deletes a session explicitly before it times out. (For more information about the
        session inactivity timeout, see the
        [documentation](https://cloud.ibm.com/docs/assistant?topic=assistant-assistant-settings)).

        :param str assistant_id: Unique identifier of the assistant. To find the
               assistant ID in the Watson Assistant user interface, open the assistant
               settings and click **API Details**. For information about creating
               assistants, see the
               [documentation](https://cloud.ibm.com/docs/assistant?topic=assistant-assistant-add#assistant-add-task).
               **Note:** Currently, the v2 API does not support creating assistants.
        :param str session_id: Unique identifier of the session.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if assistant_id is None:
            raise ValueError('assistant_id must be provided')
        if session_id is None:
            raise ValueError('session_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V2',
                                      operation_id='delete_session')
        headers.update(sdk_headers)

        params = {'version': self.version}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        path_param_keys = ['assistant_id', 'session_id']
        path_param_values = self.encode_path_vars(assistant_id, session_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/v2/assistants/{assistant_id}/sessions/{session_id}'.format(
            **path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Message
    #########################

    def message(self,
                assistant_id: str,
                session_id: str,
                *,
                input: 'MessageInput' = None,
                context: 'MessageContext' = None,
                user_id: str = None,
                **kwargs) -> DetailedResponse:
        """
        Send user input to assistant (stateful).

        Send user input to an assistant and receive a response, with conversation state
        (including context data) stored by Watson Assistant for the duration of the
        session.

        :param str assistant_id: Unique identifier of the assistant. To find the
               assistant ID in the Watson Assistant user interface, open the assistant
               settings and click **API Details**. For information about creating
               assistants, see the
               [documentation](https://cloud.ibm.com/docs/assistant?topic=assistant-assistant-add#assistant-add-task).
               **Note:** Currently, the v2 API does not support creating assistants.
        :param str session_id: Unique identifier of the session.
        :param MessageInput input: (optional) An input object that includes the
               input text.
        :param MessageContext context: (optional) Context data for the
               conversation. You can use this property to set or modify context variables,
               which can also be accessed by dialog nodes. The context is stored by the
               assistant on a per-session basis.
               **Note:** The total size of the context data stored for a stateful session
               cannot exceed 100KB.
        :param str user_id: (optional) A string value that identifies the user who
               is interacting with the assistant. The client must provide a unique
               identifier for each individual end user who accesses the application. For
               user-based plans, this user ID is used to identify unique users for billing
               purposes. This string cannot contain carriage return, newline, or tab
               characters. If no value is specified in the input, **user_id** is
               automatically set to the value of **context.global.session_id**.
               **Note:** This property is the same as the **user_id** property in the
               global system context. If **user_id** is specified in both locations, the
               value specified at the root is used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `MessageResponse` object
        """

        if assistant_id is None:
            raise ValueError('assistant_id must be provided')
        if session_id is None:
            raise ValueError('session_id must be provided')
        if input is not None:
            input = convert_model(input)
        if context is not None:
            context = convert_model(context)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V2',
                                      operation_id='message')
        headers.update(sdk_headers)

        params = {'version': self.version}

        data = {'input': input, 'context': context, 'user_id': user_id}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        path_param_keys = ['assistant_id', 'session_id']
        path_param_values = self.encode_path_vars(assistant_id, session_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/v2/assistants/{assistant_id}/sessions/{session_id}/message'.format(
            **path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    def message_stateless(self,
                          assistant_id: str,
                          *,
                          input: 'MessageInputStateless' = None,
                          context: 'MessageContextStateless' = None,
                          user_id: str = None,
                          **kwargs) -> DetailedResponse:
        """
        Send user input to assistant (stateless).

        Send user input to an assistant and receive a response, with conversation state
        (including context data) managed by your application.

        :param str assistant_id: Unique identifier of the assistant. To find the
               assistant ID in the Watson Assistant user interface, open the assistant
               settings and click **API Details**. For information about creating
               assistants, see the
               [documentation](https://cloud.ibm.com/docs/assistant?topic=assistant-assistant-add#assistant-add-task).
               **Note:** Currently, the v2 API does not support creating assistants.
        :param MessageInputStateless input: (optional) An input object that
               includes the input text.
        :param MessageContextStateless context: (optional) Context data for the
               conversation. You can use this property to set or modify context variables,
               which can also be accessed by dialog nodes. The context is not stored by
               the assistant. To maintain session state, include the context from the
               previous response.
               **Note:** The total size of the context data for a stateless session cannot
               exceed 250KB.
        :param str user_id: (optional) A string value that identifies the user who
               is interacting with the assistant. The client must provide a unique
               identifier for each individual end user who accesses the application. For
               user-based plans, this user ID is used to identify unique users for billing
               purposes. This string cannot contain carriage return, newline, or tab
               characters. If no value is specified in the input, **user_id** is
               automatically set to the value of **context.global.session_id**.
               **Note:** This property is the same as the **user_id** property in the
               global system context. If **user_id** is specified in both locations in a
               message request, the value specified at the root is used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `MessageResponseStateless` object
        """

        if assistant_id is None:
            raise ValueError('assistant_id must be provided')
        if input is not None:
            input = convert_model(input)
        if context is not None:
            context = convert_model(context)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V2',
                                      operation_id='message_stateless')
        headers.update(sdk_headers)

        params = {'version': self.version}

        data = {'input': input, 'context': context, 'user_id': user_id}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        path_param_keys = ['assistant_id']
        path_param_values = self.encode_path_vars(assistant_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/v2/assistants/{assistant_id}/message'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Bulk classify
    #########################

    def bulk_classify(self,
                      skill_id: str,
                      *,
                      input: List['BulkClassifyUtterance'] = None,
                      **kwargs) -> DetailedResponse:
        """
        Identify intents and entities in multiple user utterances.

        Send multiple user inputs to a dialog skill in a single request and receive
        information about the intents and entities recognized in each input. This method
        is useful for testing and comparing the performance of different skills or skill
        versions.
        This method is available only with Enterprise with Data Isolation plans.

        :param str skill_id: Unique identifier of the skill. To find the skill ID
               in the Watson Assistant user interface, open the skill settings and click
               **API Details**.
        :param List[BulkClassifyUtterance] input: (optional) An array of input
               utterances to classify.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BulkClassifyResponse` object
        """

        if skill_id is None:
            raise ValueError('skill_id must be provided')
        if input is not None:
            input = [convert_model(x) for x in input]
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V2',
                                      operation_id='bulk_classify')
        headers.update(sdk_headers)

        params = {'version': self.version}

        data = {'input': input}
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        path_param_keys = ['skill_id']
        path_param_values = self.encode_path_vars(skill_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/v2/skills/{skill_id}/workspace/bulk_classify'.format(
            **path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Logs
    #########################

    def list_logs(self,
                  assistant_id: str,
                  *,
                  sort: str = None,
                  filter: str = None,
                  page_limit: int = None,
                  cursor: str = None,
                  **kwargs) -> DetailedResponse:
        """
        List log events for an assistant.

        List the events from the log of an assistant.
        This method requires Manager access, and is available only with Enterprise plans.

        :param str assistant_id: Unique identifier of the assistant. To find the
               assistant ID in the Watson Assistant user interface, open the assistant
               settings and click **API Details**. For information about creating
               assistants, see the
               [documentation](https://cloud.ibm.com/docs/assistant?topic=assistant-assistant-add#assistant-add-task).
               **Note:** Currently, the v2 API does not support creating assistants.
        :param str sort: (optional) How to sort the returned log events. You can
               sort by **request_timestamp**. To reverse the sort order, prefix the
               parameter value with a minus sign (`-`).
        :param str filter: (optional) A cacheable parameter that limits the results
               to those matching the specified filter. For more information, see the
               [documentation](https://cloud.ibm.com/docs/assistant?topic=assistant-filter-reference#filter-reference).
        :param int page_limit: (optional) The number of records to return in each
               page of results.
        :param str cursor: (optional) A token identifying the page of results to
               retrieve.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LogCollection` object
        """

        if assistant_id is None:
            raise ValueError('assistant_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V2',
                                      operation_id='list_logs')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'sort': sort,
            'filter': filter,
            'page_limit': page_limit,
            'cursor': cursor
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        path_param_keys = ['assistant_id']
        path_param_values = self.encode_path_vars(assistant_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/v2/assistants/{assistant_id}/logs'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response

    #########################
    # User data
    #########################

    def delete_user_data(self, customer_id: str, **kwargs) -> DetailedResponse:
        """
        Delete labeled data.

        Deletes all data associated with a specified customer ID. The method has no effect
        if no data is associated with the customer ID.
        You associate a customer ID with data by passing the `X-Watson-Metadata` header
        with a request that passes data. For more information about personal data and
        customer IDs, see [Information
        security](https://cloud.ibm.com/docs/assistant?topic=assistant-information-security#information-security).
        **Note:** This operation is intended only for deleting data associated with a
        single specific customer, not for deleting data associated with multiple customers
        or for any other purpose. For more information, see [Labeling and deleting data in
        Watson
        Assistant](https://cloud.ibm.com/docs/assistant?topic=assistant-information-security#information-security-gdpr-wa).

        :param str customer_id: The customer ID for which all data is to be
               deleted.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if customer_id is None:
            raise ValueError('customer_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V2',
                                      operation_id='delete_user_data')
        headers.update(sdk_headers)

        params = {'version': self.version, 'customer_id': customer_id}

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
        headers['Accept'] = 'application/json'

        url = '/v2/user_data'
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


##############################################################################
# Models
##############################################################################


class AgentAvailabilityMessage():
    """
    AgentAvailabilityMessage.

    :attr str message: (optional) The text of the message.
    """

    def __init__(self, *, message: str = None) -> None:
        """
        Initialize a AgentAvailabilityMessage object.

        :param str message: (optional) The text of the message.
        """
        self.message = message

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'AgentAvailabilityMessage':
        """Initialize a AgentAvailabilityMessage object from a json dictionary."""
        args = {}
        if 'message' in _dict:
            args['message'] = _dict.get('message')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a AgentAvailabilityMessage object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'message') and self.message is not None:
            _dict['message'] = self.message
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this AgentAvailabilityMessage object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'AgentAvailabilityMessage') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'AgentAvailabilityMessage') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class BulkClassifyOutput():
    """
    BulkClassifyOutput.

    :attr BulkClassifyUtterance input: (optional) The user input utterance to
          classify.
    :attr List[RuntimeEntity] entities: (optional) An array of entities identified
          in the utterance.
    :attr List[RuntimeIntent] intents: (optional) An array of intents recognized in
          the utterance.
    """

    def __init__(self,
                 *,
                 input: 'BulkClassifyUtterance' = None,
                 entities: List['RuntimeEntity'] = None,
                 intents: List['RuntimeIntent'] = None) -> None:
        """
        Initialize a BulkClassifyOutput object.

        :param BulkClassifyUtterance input: (optional) The user input utterance to
               classify.
        :param List[RuntimeEntity] entities: (optional) An array of entities
               identified in the utterance.
        :param List[RuntimeIntent] intents: (optional) An array of intents
               recognized in the utterance.
        """
        self.input = input
        self.entities = entities
        self.intents = intents

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BulkClassifyOutput':
        """Initialize a BulkClassifyOutput object from a json dictionary."""
        args = {}
        if 'input' in _dict:
            args['input'] = BulkClassifyUtterance.from_dict(_dict.get('input'))
        if 'entities' in _dict:
            args['entities'] = [
                RuntimeEntity.from_dict(x) for x in _dict.get('entities')
            ]
        if 'intents' in _dict:
            args['intents'] = [
                RuntimeIntent.from_dict(x) for x in _dict.get('intents')
            ]
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BulkClassifyOutput object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'input') and self.input is not None:
            _dict['input'] = self.input.to_dict()
        if hasattr(self, 'entities') and self.entities is not None:
            _dict['entities'] = [x.to_dict() for x in self.entities]
        if hasattr(self, 'intents') and self.intents is not None:
            _dict['intents'] = [x.to_dict() for x in self.intents]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BulkClassifyOutput object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BulkClassifyOutput') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BulkClassifyOutput') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class BulkClassifyResponse():
    """
    BulkClassifyResponse.

    :attr List[BulkClassifyOutput] output: (optional) An array of objects that
          contain classification information for the submitted input utterances.
    """

    def __init__(self, *, output: List['BulkClassifyOutput'] = None) -> None:
        """
        Initialize a BulkClassifyResponse object.

        :param List[BulkClassifyOutput] output: (optional) An array of objects that
               contain classification information for the submitted input utterances.
        """
        self.output = output

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BulkClassifyResponse':
        """Initialize a BulkClassifyResponse object from a json dictionary."""
        args = {}
        if 'output' in _dict:
            args['output'] = [
                BulkClassifyOutput.from_dict(x) for x in _dict.get('output')
            ]
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BulkClassifyResponse object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'output') and self.output is not None:
            _dict['output'] = [x.to_dict() for x in self.output]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BulkClassifyResponse object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BulkClassifyResponse') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BulkClassifyResponse') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class BulkClassifyUtterance():
    """
    The user input utterance to classify.

    :attr str text: The text of the input utterance.
    """

    def __init__(self, text: str) -> None:
        """
        Initialize a BulkClassifyUtterance object.

        :param str text: The text of the input utterance.
        """
        self.text = text

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BulkClassifyUtterance':
        """Initialize a BulkClassifyUtterance object from a json dictionary."""
        args = {}
        if 'text' in _dict:
            args['text'] = _dict.get('text')
        else:
            raise ValueError(
                'Required property \'text\' not present in BulkClassifyUtterance JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BulkClassifyUtterance object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'text') and self.text is not None:
            _dict['text'] = self.text
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BulkClassifyUtterance object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BulkClassifyUtterance') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BulkClassifyUtterance') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class CaptureGroup():
    """
    CaptureGroup.

    :attr str group: A recognized capture group for the entity.
    :attr List[int] location: (optional) Zero-based character offsets that indicate
          where the entity value begins and ends in the input text.
    """

    def __init__(self, group: str, *, location: List[int] = None) -> None:
        """
        Initialize a CaptureGroup object.

        :param str group: A recognized capture group for the entity.
        :param List[int] location: (optional) Zero-based character offsets that
               indicate where the entity value begins and ends in the input text.
        """
        self.group = group
        self.location = location

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'CaptureGroup':
        """Initialize a CaptureGroup object from a json dictionary."""
        args = {}
        if 'group' in _dict:
            args['group'] = _dict.get('group')
        else:
            raise ValueError(
                'Required property \'group\' not present in CaptureGroup JSON')
        if 'location' in _dict:
            args['location'] = _dict.get('location')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a CaptureGroup object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'group') and self.group is not None:
            _dict['group'] = self.group
        if hasattr(self, 'location') and self.location is not None:
            _dict['location'] = self.location
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this CaptureGroup object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'CaptureGroup') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'CaptureGroup') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class ChannelTransferInfo():
    """
    Information used by an integration to transfer the conversation to a different
    channel.

    :attr ChannelTransferTarget target: An object specifying target channels
          available for the transfer. Each property of this object represents an available
          transfer target. Currently, the only supported property is **chat**,
          representing the web chat integration.
    """

    def __init__(self, target: 'ChannelTransferTarget') -> None:
        """
        Initialize a ChannelTransferInfo object.

        :param ChannelTransferTarget target: An object specifying target channels
               available for the transfer. Each property of this object represents an
               available transfer target. Currently, the only supported property is
               **chat**, representing the web chat integration.
        """
        self.target = target

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ChannelTransferInfo':
        """Initialize a ChannelTransferInfo object from a json dictionary."""
        args = {}
        if 'target' in _dict:
            args['target'] = ChannelTransferTarget.from_dict(
                _dict.get('target'))
        else:
            raise ValueError(
                'Required property \'target\' not present in ChannelTransferInfo JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ChannelTransferInfo object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'target') and self.target is not None:
            _dict['target'] = self.target.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ChannelTransferInfo object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ChannelTransferInfo') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ChannelTransferInfo') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class ChannelTransferTarget():
    """
    An object specifying target channels available for the transfer. Each property of this
    object represents an available transfer target. Currently, the only supported property
    is **chat**, representing the web chat integration.

    :attr ChannelTransferTargetChat chat: (optional) Information for transferring to
          the web chat integration.
    """

    def __init__(self, *, chat: 'ChannelTransferTargetChat' = None) -> None:
        """
        Initialize a ChannelTransferTarget object.

        :param ChannelTransferTargetChat chat: (optional) Information for
               transferring to the web chat integration.
        """
        self.chat = chat

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ChannelTransferTarget':
        """Initialize a ChannelTransferTarget object from a json dictionary."""
        args = {}
        if 'chat' in _dict:
            args['chat'] = ChannelTransferTargetChat.from_dict(
                _dict.get('chat'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ChannelTransferTarget object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'chat') and self.chat is not None:
            _dict['chat'] = self.chat.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ChannelTransferTarget object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ChannelTransferTarget') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ChannelTransferTarget') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class ChannelTransferTargetChat():
    """
    Information for transferring to the web chat integration.

    :attr str url: (optional) The URL of the target web chat.
    """

    def __init__(self, *, url: str = None) -> None:
        """
        Initialize a ChannelTransferTargetChat object.

        :param str url: (optional) The URL of the target web chat.
        """
        self.url = url

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ChannelTransferTargetChat':
        """Initialize a ChannelTransferTargetChat object from a json dictionary."""
        args = {}
        if 'url' in _dict:
            args['url'] = _dict.get('url')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ChannelTransferTargetChat object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'url') and self.url is not None:
            _dict['url'] = self.url
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ChannelTransferTargetChat object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ChannelTransferTargetChat') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ChannelTransferTargetChat') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class DialogLogMessage():
    """
    Dialog log message details.

    :attr str level: The severity of the log message.
    :attr str message: The text of the log message.
    :attr str code: A code that indicates the category to which the error message
          belongs.
    :attr LogMessageSource source: (optional) An object that identifies the dialog
          element that generated the error message.
    """

    def __init__(self,
                 level: str,
                 message: str,
                 code: str,
                 *,
                 source: 'LogMessageSource' = None) -> None:
        """
        Initialize a DialogLogMessage object.

        :param str level: The severity of the log message.
        :param str message: The text of the log message.
        :param str code: A code that indicates the category to which the error
               message belongs.
        :param LogMessageSource source: (optional) An object that identifies the
               dialog element that generated the error message.
        """
        self.level = level
        self.message = message
        self.code = code
        self.source = source

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'DialogLogMessage':
        """Initialize a DialogLogMessage object from a json dictionary."""
        args = {}
        if 'level' in _dict:
            args['level'] = _dict.get('level')
        else:
            raise ValueError(
                'Required property \'level\' not present in DialogLogMessage JSON'
            )
        if 'message' in _dict:
            args['message'] = _dict.get('message')
        else:
            raise ValueError(
                'Required property \'message\' not present in DialogLogMessage JSON'
            )
        if 'code' in _dict:
            args['code'] = _dict.get('code')
        else:
            raise ValueError(
                'Required property \'code\' not present in DialogLogMessage JSON'
            )
        if 'source' in _dict:
            args['source'] = LogMessageSource.from_dict(_dict.get('source'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a DialogLogMessage object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'level') and self.level is not None:
            _dict['level'] = self.level
        if hasattr(self, 'message') and self.message is not None:
            _dict['message'] = self.message
        if hasattr(self, 'code') and self.code is not None:
            _dict['code'] = self.code
        if hasattr(self, 'source') and self.source is not None:
            _dict['source'] = self.source.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this DialogLogMessage object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'DialogLogMessage') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'DialogLogMessage') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class LevelEnum(str, Enum):
        """
        The severity of the log message.
        """
        INFO = 'info'
        ERROR = 'error'
        WARN = 'warn'


class DialogNodeAction():
    """
    DialogNodeAction.

    :attr str name: The name of the action.
    :attr str type: (optional) The type of action to invoke.
    :attr dict parameters: (optional) A map of key/value pairs to be provided to the
          action.
    :attr str result_variable: The location in the dialog context where the result
          of the action is stored.
    :attr str credentials: (optional) The name of the context variable that the
          client application will use to pass in credentials for the action.
    """

    def __init__(self,
                 name: str,
                 result_variable: str,
                 *,
                 type: str = None,
                 parameters: dict = None,
                 credentials: str = None) -> None:
        """
        Initialize a DialogNodeAction object.

        :param str name: The name of the action.
        :param str result_variable: The location in the dialog context where the
               result of the action is stored.
        :param str type: (optional) The type of action to invoke.
        :param dict parameters: (optional) A map of key/value pairs to be provided
               to the action.
        :param str credentials: (optional) The name of the context variable that
               the client application will use to pass in credentials for the action.
        """
        self.name = name
        self.type = type
        self.parameters = parameters
        self.result_variable = result_variable
        self.credentials = credentials

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'DialogNodeAction':
        """Initialize a DialogNodeAction object from a json dictionary."""
        args = {}
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError(
                'Required property \'name\' not present in DialogNodeAction JSON'
            )
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        if 'parameters' in _dict:
            args['parameters'] = _dict.get('parameters')
        if 'result_variable' in _dict:
            args['result_variable'] = _dict.get('result_variable')
        else:
            raise ValueError(
                'Required property \'result_variable\' not present in DialogNodeAction JSON'
            )
        if 'credentials' in _dict:
            args['credentials'] = _dict.get('credentials')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a DialogNodeAction object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        if hasattr(self, 'parameters') and self.parameters is not None:
            _dict['parameters'] = self.parameters
        if hasattr(self,
                   'result_variable') and self.result_variable is not None:
            _dict['result_variable'] = self.result_variable
        if hasattr(self, 'credentials') and self.credentials is not None:
            _dict['credentials'] = self.credentials
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this DialogNodeAction object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'DialogNodeAction') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'DialogNodeAction') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class TypeEnum(str, Enum):
        """
        The type of action to invoke.
        """
        CLIENT = 'client'
        SERVER = 'server'
        WEB_ACTION = 'web-action'
        CLOUD_FUNCTION = 'cloud-function'


class DialogNodeOutputConnectToAgentTransferInfo():
    """
    Routing or other contextual information to be used by target service desk systems.

    :attr dict target: (optional)
    """

    def __init__(self, *, target: dict = None) -> None:
        """
        Initialize a DialogNodeOutputConnectToAgentTransferInfo object.

        :param dict target: (optional)
        """
        self.target = target

    @classmethod
    def from_dict(cls,
                  _dict: Dict) -> 'DialogNodeOutputConnectToAgentTransferInfo':
        """Initialize a DialogNodeOutputConnectToAgentTransferInfo object from a json dictionary."""
        args = {}
        if 'target' in _dict:
            args['target'] = _dict.get('target')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a DialogNodeOutputConnectToAgentTransferInfo object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'target') and self.target is not None:
            _dict['target'] = self.target
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this DialogNodeOutputConnectToAgentTransferInfo object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self,
               other: 'DialogNodeOutputConnectToAgentTransferInfo') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self,
               other: 'DialogNodeOutputConnectToAgentTransferInfo') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class DialogNodeOutputOptionsElement():
    """
    DialogNodeOutputOptionsElement.

    :attr str label: The user-facing label for the option.
    :attr DialogNodeOutputOptionsElementValue value: An object defining the message
          input to be sent to the assistant if the user selects the corresponding option.
    """

    def __init__(self, label: str,
                 value: 'DialogNodeOutputOptionsElementValue') -> None:
        """
        Initialize a DialogNodeOutputOptionsElement object.

        :param str label: The user-facing label for the option.
        :param DialogNodeOutputOptionsElementValue value: An object defining the
               message input to be sent to the assistant if the user selects the
               corresponding option.
        """
        self.label = label
        self.value = value

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'DialogNodeOutputOptionsElement':
        """Initialize a DialogNodeOutputOptionsElement object from a json dictionary."""
        args = {}
        if 'label' in _dict:
            args['label'] = _dict.get('label')
        else:
            raise ValueError(
                'Required property \'label\' not present in DialogNodeOutputOptionsElement JSON'
            )
        if 'value' in _dict:
            args['value'] = DialogNodeOutputOptionsElementValue.from_dict(
                _dict.get('value'))
        else:
            raise ValueError(
                'Required property \'value\' not present in DialogNodeOutputOptionsElement JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a DialogNodeOutputOptionsElement object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'label') and self.label is not None:
            _dict['label'] = self.label
        if hasattr(self, 'value') and self.value is not None:
            _dict['value'] = self.value.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this DialogNodeOutputOptionsElement object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'DialogNodeOutputOptionsElement') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'DialogNodeOutputOptionsElement') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class DialogNodeOutputOptionsElementValue():
    """
    An object defining the message input to be sent to the assistant if the user selects
    the corresponding option.

    :attr MessageInput input: (optional) An input object that includes the input
          text.
    """

    def __init__(self, *, input: 'MessageInput' = None) -> None:
        """
        Initialize a DialogNodeOutputOptionsElementValue object.

        :param MessageInput input: (optional) An input object that includes the
               input text.
        """
        self.input = input

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'DialogNodeOutputOptionsElementValue':
        """Initialize a DialogNodeOutputOptionsElementValue object from a json dictionary."""
        args = {}
        if 'input' in _dict:
            args['input'] = MessageInput.from_dict(_dict.get('input'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a DialogNodeOutputOptionsElementValue object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'input') and self.input is not None:
            _dict['input'] = self.input.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this DialogNodeOutputOptionsElementValue object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'DialogNodeOutputOptionsElementValue') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'DialogNodeOutputOptionsElementValue') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class DialogNodeVisited():
    """
    An objects containing detailed diagnostic information about a dialog node that was
    triggered during processing of the input message.

    :attr str dialog_node: (optional) A dialog node that was triggered during
          processing of the input message.
    :attr str title: (optional) The title of the dialog node.
    :attr str conditions: (optional) The conditions that trigger the dialog node.
    """

    def __init__(self,
                 *,
                 dialog_node: str = None,
                 title: str = None,
                 conditions: str = None) -> None:
        """
        Initialize a DialogNodeVisited object.

        :param str dialog_node: (optional) A dialog node that was triggered during
               processing of the input message.
        :param str title: (optional) The title of the dialog node.
        :param str conditions: (optional) The conditions that trigger the dialog
               node.
        """
        self.dialog_node = dialog_node
        self.title = title
        self.conditions = conditions

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'DialogNodeVisited':
        """Initialize a DialogNodeVisited object from a json dictionary."""
        args = {}
        if 'dialog_node' in _dict:
            args['dialog_node'] = _dict.get('dialog_node')
        if 'title' in _dict:
            args['title'] = _dict.get('title')
        if 'conditions' in _dict:
            args['conditions'] = _dict.get('conditions')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a DialogNodeVisited object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'dialog_node') and self.dialog_node is not None:
            _dict['dialog_node'] = self.dialog_node
        if hasattr(self, 'title') and self.title is not None:
            _dict['title'] = self.title
        if hasattr(self, 'conditions') and self.conditions is not None:
            _dict['conditions'] = self.conditions
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this DialogNodeVisited object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'DialogNodeVisited') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'DialogNodeVisited') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class DialogSuggestion():
    """
    DialogSuggestion.

    :attr str label: The user-facing label for the suggestion. This label is taken
          from the **title** or **user_label** property of the corresponding dialog node,
          depending on the disambiguation options.
    :attr DialogSuggestionValue value: An object defining the message input to be
          sent to the assistant if the user selects the corresponding disambiguation
          option.
    :attr dict output: (optional) The dialog output that will be returned from the
          Watson Assistant service if the user selects the corresponding option.
    """

    def __init__(self,
                 label: str,
                 value: 'DialogSuggestionValue',
                 *,
                 output: dict = None) -> None:
        """
        Initialize a DialogSuggestion object.

        :param str label: The user-facing label for the suggestion. This label is
               taken from the **title** or **user_label** property of the corresponding
               dialog node, depending on the disambiguation options.
        :param DialogSuggestionValue value: An object defining the message input to
               be sent to the assistant if the user selects the corresponding
               disambiguation option.
        :param dict output: (optional) The dialog output that will be returned from
               the Watson Assistant service if the user selects the corresponding option.
        """
        self.label = label
        self.value = value
        self.output = output

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'DialogSuggestion':
        """Initialize a DialogSuggestion object from a json dictionary."""
        args = {}
        if 'label' in _dict:
            args['label'] = _dict.get('label')
        else:
            raise ValueError(
                'Required property \'label\' not present in DialogSuggestion JSON'
            )
        if 'value' in _dict:
            args['value'] = DialogSuggestionValue.from_dict(_dict.get('value'))
        else:
            raise ValueError(
                'Required property \'value\' not present in DialogSuggestion JSON'
            )
        if 'output' in _dict:
            args['output'] = _dict.get('output')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a DialogSuggestion object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'label') and self.label is not None:
            _dict['label'] = self.label
        if hasattr(self, 'value') and self.value is not None:
            _dict['value'] = self.value.to_dict()
        if hasattr(self, 'output') and self.output is not None:
            _dict['output'] = self.output
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this DialogSuggestion object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'DialogSuggestion') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'DialogSuggestion') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class DialogSuggestionValue():
    """
    An object defining the message input to be sent to the assistant if the user selects
    the corresponding disambiguation option.

    :attr MessageInput input: (optional) An input object that includes the input
          text.
    """

    def __init__(self, *, input: 'MessageInput' = None) -> None:
        """
        Initialize a DialogSuggestionValue object.

        :param MessageInput input: (optional) An input object that includes the
               input text.
        """
        self.input = input

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'DialogSuggestionValue':
        """Initialize a DialogSuggestionValue object from a json dictionary."""
        args = {}
        if 'input' in _dict:
            args['input'] = MessageInput.from_dict(_dict.get('input'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a DialogSuggestionValue object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'input') and self.input is not None:
            _dict['input'] = self.input.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this DialogSuggestionValue object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'DialogSuggestionValue') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'DialogSuggestionValue') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class Log():
    """
    Log.

    :attr str log_id: A unique identifier for the logged event.
    :attr MessageRequest request: A stateful message request formatted for the
          Watson Assistant service.
    :attr MessageResponse response: A response from the Watson Assistant service.
    :attr str assistant_id: Unique identifier of the assistant.
    :attr str session_id: The ID of the session the message was part of.
    :attr str skill_id: The unique identifier of the skill that responded to the
          message.
    :attr str snapshot: The name of the snapshot (dialog skill version) that
          responded to the message (for example, `draft`).
    :attr str request_timestamp: The timestamp for receipt of the message.
    :attr str response_timestamp: The timestamp for the system response to the
          message.
    :attr str language: The language of the assistant to which the message request
          was made.
    :attr str customer_id: (optional) The customer ID specified for the message, if
          any.
    """

    def __init__(self,
                 log_id: str,
                 request: 'MessageRequest',
                 response: 'MessageResponse',
                 assistant_id: str,
                 session_id: str,
                 skill_id: str,
                 snapshot: str,
                 request_timestamp: str,
                 response_timestamp: str,
                 language: str,
                 *,
                 customer_id: str = None) -> None:
        """
        Initialize a Log object.

        :param str log_id: A unique identifier for the logged event.
        :param MessageRequest request: A stateful message request formatted for the
               Watson Assistant service.
        :param MessageResponse response: A response from the Watson Assistant
               service.
        :param str assistant_id: Unique identifier of the assistant.
        :param str session_id: The ID of the session the message was part of.
        :param str skill_id: The unique identifier of the skill that responded to
               the message.
        :param str snapshot: The name of the snapshot (dialog skill version) that
               responded to the message (for example, `draft`).
        :param str request_timestamp: The timestamp for receipt of the message.
        :param str response_timestamp: The timestamp for the system response to the
               message.
        :param str language: The language of the assistant to which the message
               request was made.
        :param str customer_id: (optional) The customer ID specified for the
               message, if any.
        """
        self.log_id = log_id
        self.request = request
        self.response = response
        self.assistant_id = assistant_id
        self.session_id = session_id
        self.skill_id = skill_id
        self.snapshot = snapshot
        self.request_timestamp = request_timestamp
        self.response_timestamp = response_timestamp
        self.language = language
        self.customer_id = customer_id

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'Log':
        """Initialize a Log object from a json dictionary."""
        args = {}
        if 'log_id' in _dict:
            args['log_id'] = _dict.get('log_id')
        else:
            raise ValueError(
                'Required property \'log_id\' not present in Log JSON')
        if 'request' in _dict:
            args['request'] = MessageRequest.from_dict(_dict.get('request'))
        else:
            raise ValueError(
                'Required property \'request\' not present in Log JSON')
        if 'response' in _dict:
            args['response'] = MessageResponse.from_dict(_dict.get('response'))
        else:
            raise ValueError(
                'Required property \'response\' not present in Log JSON')
        if 'assistant_id' in _dict:
            args['assistant_id'] = _dict.get('assistant_id')
        else:
            raise ValueError(
                'Required property \'assistant_id\' not present in Log JSON')
        if 'session_id' in _dict:
            args['session_id'] = _dict.get('session_id')
        else:
            raise ValueError(
                'Required property \'session_id\' not present in Log JSON')
        if 'skill_id' in _dict:
            args['skill_id'] = _dict.get('skill_id')
        else:
            raise ValueError(
                'Required property \'skill_id\' not present in Log JSON')
        if 'snapshot' in _dict:
            args['snapshot'] = _dict.get('snapshot')
        else:
            raise ValueError(
                'Required property \'snapshot\' not present in Log JSON')
        if 'request_timestamp' in _dict:
            args['request_timestamp'] = _dict.get('request_timestamp')
        else:
            raise ValueError(
                'Required property \'request_timestamp\' not present in Log JSON'
            )
        if 'response_timestamp' in _dict:
            args['response_timestamp'] = _dict.get('response_timestamp')
        else:
            raise ValueError(
                'Required property \'response_timestamp\' not present in Log JSON'
            )
        if 'language' in _dict:
            args['language'] = _dict.get('language')
        else:
            raise ValueError(
                'Required property \'language\' not present in Log JSON')
        if 'customer_id' in _dict:
            args['customer_id'] = _dict.get('customer_id')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a Log object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'log_id') and self.log_id is not None:
            _dict['log_id'] = self.log_id
        if hasattr(self, 'request') and self.request is not None:
            _dict['request'] = self.request.to_dict()
        if hasattr(self, 'response') and self.response is not None:
            _dict['response'] = self.response.to_dict()
        if hasattr(self, 'assistant_id') and self.assistant_id is not None:
            _dict['assistant_id'] = self.assistant_id
        if hasattr(self, 'session_id') and self.session_id is not None:
            _dict['session_id'] = self.session_id
        if hasattr(self, 'skill_id') and self.skill_id is not None:
            _dict['skill_id'] = self.skill_id
        if hasattr(self, 'snapshot') and self.snapshot is not None:
            _dict['snapshot'] = self.snapshot
        if hasattr(self,
                   'request_timestamp') and self.request_timestamp is not None:
            _dict['request_timestamp'] = self.request_timestamp
        if hasattr(
                self,
                'response_timestamp') and self.response_timestamp is not None:
            _dict['response_timestamp'] = self.response_timestamp
        if hasattr(self, 'language') and self.language is not None:
            _dict['language'] = self.language
        if hasattr(self, 'customer_id') and self.customer_id is not None:
            _dict['customer_id'] = self.customer_id
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this Log object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'Log') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'Log') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LogCollection():
    """
    LogCollection.

    :attr List[Log] logs: An array of objects describing log events.
    :attr LogPagination pagination: The pagination data for the returned objects.
    """

    def __init__(self, logs: List['Log'], pagination: 'LogPagination') -> None:
        """
        Initialize a LogCollection object.

        :param List[Log] logs: An array of objects describing log events.
        :param LogPagination pagination: The pagination data for the returned
               objects.
        """
        self.logs = logs
        self.pagination = pagination

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LogCollection':
        """Initialize a LogCollection object from a json dictionary."""
        args = {}
        if 'logs' in _dict:
            args['logs'] = [Log.from_dict(x) for x in _dict.get('logs')]
        else:
            raise ValueError(
                'Required property \'logs\' not present in LogCollection JSON')
        if 'pagination' in _dict:
            args['pagination'] = LogPagination.from_dict(
                _dict.get('pagination'))
        else:
            raise ValueError(
                'Required property \'pagination\' not present in LogCollection JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LogCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'logs') and self.logs is not None:
            _dict['logs'] = [x.to_dict() for x in self.logs]
        if hasattr(self, 'pagination') and self.pagination is not None:
            _dict['pagination'] = self.pagination.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LogCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LogCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LogCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LogMessageSource():
    """
    An object that identifies the dialog element that generated the error message.

    """

    def __init__(self) -> None:
        """
        Initialize a LogMessageSource object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'LogMessageSourceDialogNode', 'LogMessageSourceAction',
                'LogMessageSourceStep', 'LogMessageSourceHandler'
            ]))
        raise Exception(msg)

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LogMessageSource':
        """Initialize a LogMessageSource object from a json dictionary."""
        disc_class = cls._get_class_by_discriminator(_dict)
        if disc_class != cls:
            return disc_class.from_dict(_dict)
        msg = (
            "Cannot convert dictionary into an instance of base class 'LogMessageSource'. "
            + "The discriminator value should map to a valid subclass: {1}"
        ).format(", ".join([
            'LogMessageSourceDialogNode', 'LogMessageSourceAction',
            'LogMessageSourceStep', 'LogMessageSourceHandler'
        ]))
        raise Exception(msg)

    @classmethod
    def _from_dict(cls, _dict: Dict):
        """Initialize a LogMessageSource object from a json dictionary."""
        return cls.from_dict(_dict)

    @classmethod
    def _get_class_by_discriminator(cls, _dict: Dict) -> object:
        mapping = {}
        mapping['dialog_node'] = 'LogMessageSourceDialogNode'
        mapping['action'] = 'LogMessageSourceAction'
        mapping['step'] = 'LogMessageSourceStep'
        mapping['handler'] = 'LogMessageSourceHandler'
        disc_value = _dict.get('type')
        if disc_value is None:
            raise ValueError(
                'Discriminator property \'type\' not found in LogMessageSource JSON'
            )
        class_name = mapping.get(disc_value, disc_value)
        try:
            disc_class = getattr(sys.modules[__name__], class_name)
        except AttributeError:
            disc_class = cls
        if isinstance(disc_class, object):
            return disc_class
        raise TypeError('%s is not a discriminator class' % class_name)


class LogPagination():
    """
    The pagination data for the returned objects.

    :attr str next_url: (optional) The URL that will return the next page of
          results, if any.
    :attr int matched: (optional) Reserved for future use.
    :attr str next_cursor: (optional) A token identifying the next page of results.
    """

    def __init__(self,
                 *,
                 next_url: str = None,
                 matched: int = None,
                 next_cursor: str = None) -> None:
        """
        Initialize a LogPagination object.

        :param str next_url: (optional) The URL that will return the next page of
               results, if any.
        :param int matched: (optional) Reserved for future use.
        :param str next_cursor: (optional) A token identifying the next page of
               results.
        """
        self.next_url = next_url
        self.matched = matched
        self.next_cursor = next_cursor

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LogPagination':
        """Initialize a LogPagination object from a json dictionary."""
        args = {}
        if 'next_url' in _dict:
            args['next_url'] = _dict.get('next_url')
        if 'matched' in _dict:
            args['matched'] = _dict.get('matched')
        if 'next_cursor' in _dict:
            args['next_cursor'] = _dict.get('next_cursor')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LogPagination object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'next_url') and self.next_url is not None:
            _dict['next_url'] = self.next_url
        if hasattr(self, 'matched') and self.matched is not None:
            _dict['matched'] = self.matched
        if hasattr(self, 'next_cursor') and self.next_cursor is not None:
            _dict['next_cursor'] = self.next_cursor
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LogPagination object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LogPagination') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LogPagination') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class MessageContext():
    """
    MessageContext.

    :attr MessageContextGlobal global_: (optional) Session context data that is
          shared by all skills used by the assistant.
    :attr dict skills: (optional) Information specific to particular skills used by
          the assistant.
          **Note:** Currently, only a single child property is supported, containing
          variables that apply to the dialog skill used by the assistant.
    :attr object integrations: (optional) An object containing context data that is
          specific to particular integrations. For more information, see the
          [documentation](https://cloud.ibm.com/docs/assistant?topic=assistant-dialog-integrations).
    """

    def __init__(self,
                 *,
                 global_: 'MessageContextGlobal' = None,
                 skills: dict = None,
                 integrations: object = None) -> None:
        """
        Initialize a MessageContext object.

        :param MessageContextGlobal global_: (optional) Session context data that
               is shared by all skills used by the assistant.
        :param dict skills: (optional) Information specific to particular skills
               used by the assistant.
               **Note:** Currently, only a single child property is supported, containing
               variables that apply to the dialog skill used by the assistant.
        :param object integrations: (optional) An object containing context data
               that is specific to particular integrations. For more information, see the
               [documentation](https://cloud.ibm.com/docs/assistant?topic=assistant-dialog-integrations).
        """
        self.global_ = global_
        self.skills = skills
        self.integrations = integrations

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageContext':
        """Initialize a MessageContext object from a json dictionary."""
        args = {}
        if 'global' in _dict:
            args['global_'] = MessageContextGlobal.from_dict(
                _dict.get('global'))
        if 'skills' in _dict:
            args['skills'] = {
                k: MessageContextSkill.from_dict(v)
                for k, v in _dict.get('skills').items()
            }
        if 'integrations' in _dict:
            args['integrations'] = _dict.get('integrations')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageContext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'global_') and self.global_ is not None:
            _dict['global'] = self.global_.to_dict()
        if hasattr(self, 'skills') and self.skills is not None:
            _dict['skills'] = {k: v.to_dict() for k, v in self.skills.items()}
        if hasattr(self, 'integrations') and self.integrations is not None:
            _dict['integrations'] = self.integrations
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageContext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageContext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageContext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class MessageContextGlobal():
    """
    Session context data that is shared by all skills used by the assistant.

    :attr MessageContextGlobalSystem system: (optional) Built-in system properties
          that apply to all skills used by the assistant.
    :attr str session_id: (optional) The session ID.
    """

    def __init__(self,
                 *,
                 system: 'MessageContextGlobalSystem' = None,
                 session_id: str = None) -> None:
        """
        Initialize a MessageContextGlobal object.

        :param MessageContextGlobalSystem system: (optional) Built-in system
               properties that apply to all skills used by the assistant.
        """
        self.system = system
        self.session_id = session_id

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageContextGlobal':
        """Initialize a MessageContextGlobal object from a json dictionary."""
        args = {}
        if 'system' in _dict:
            args['system'] = MessageContextGlobalSystem.from_dict(
                _dict.get('system'))
        if 'session_id' in _dict:
            args['session_id'] = _dict.get('session_id')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageContextGlobal object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'system') and self.system is not None:
            _dict['system'] = self.system.to_dict()
        if hasattr(self, 'session_id') and getattr(self,
                                                   'session_id') is not None:
            _dict['session_id'] = getattr(self, 'session_id')
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageContextGlobal object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageContextGlobal') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageContextGlobal') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class MessageContextGlobalStateless():
    """
    Session context data that is shared by all skills used by the assistant.

    :attr MessageContextGlobalSystem system: (optional) Built-in system properties
          that apply to all skills used by the assistant.
    :attr str session_id: (optional) The unique identifier of the session.
    """

    def __init__(self,
                 *,
                 system: 'MessageContextGlobalSystem' = None,
                 session_id: str = None) -> None:
        """
        Initialize a MessageContextGlobalStateless object.

        :param MessageContextGlobalSystem system: (optional) Built-in system
               properties that apply to all skills used by the assistant.
        :param str session_id: (optional) The unique identifier of the session.
        """
        self.system = system
        self.session_id = session_id

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageContextGlobalStateless':
        """Initialize a MessageContextGlobalStateless object from a json dictionary."""
        args = {}
        if 'system' in _dict:
            args['system'] = MessageContextGlobalSystem.from_dict(
                _dict.get('system'))
        if 'session_id' in _dict:
            args['session_id'] = _dict.get('session_id')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageContextGlobalStateless object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'system') and self.system is not None:
            _dict['system'] = self.system.to_dict()
        if hasattr(self, 'session_id') and self.session_id is not None:
            _dict['session_id'] = self.session_id
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageContextGlobalStateless object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageContextGlobalStateless') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageContextGlobalStateless') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class MessageContextGlobalSystem():
    """
    Built-in system properties that apply to all skills used by the assistant.

    :attr str timezone: (optional) The user time zone. The assistant uses the time
          zone to correctly resolve relative time references.
    :attr str user_id: (optional) A string value that identifies the user who is
          interacting with the assistant. The client must provide a unique identifier for
          each individual end user who accesses the application. For user-based plans,
          this user ID is used to identify unique users for billing purposes. This string
          cannot contain carriage return, newline, or tab characters. If no value is
          specified in the input, **user_id** is automatically set to the value of
          **context.global.session_id**.
          **Note:** This property is the same as the **user_id** property at the root of
          the message body. If **user_id** is specified in both locations in a message
          request, the value specified at the root is used.
    :attr int turn_count: (optional) A counter that is automatically incremented
          with each turn of the conversation. A value of 1 indicates that this is the the
          first turn of a new conversation, which can affect the behavior of some skills
          (for example, triggering the start node of a dialog).
    :attr str locale: (optional) The language code for localization in the user
          input. The specified locale overrides the default for the assistant, and is used
          for interpreting entity values in user input such as date values. For example,
          `04/03/2018` might be interpreted either as April 3 or March 4, depending on the
          locale.
           This property is included only if the new system entities are enabled for the
          skill.
    :attr str reference_time: (optional) The base time for interpreting any relative
          time mentions in the user input. The specified time overrides the current server
          time, and is used to calculate times mentioned in relative terms such as `now`
          or `tomorrow`. This can be useful for simulating past or future times for
          testing purposes, or when analyzing documents such as news articles.
          This value must be a UTC time value formatted according to ISO 8601 (for
          example, `2021-06-26T12:00:00Z` for noon UTC on 26 June 2021).
          This property is included only if the new system entities are enabled for the
          skill.
    :attr str session_start_time: (optional) The time at which the session started.
          With the stateful `message` method, the start time is always present, and is set
          by the service based on the time the session was created. With the stateless
          `message` method, the start time is set by the service in the response to the
          first message, and should be returned as part of the context with each
          subsequent message in the session.
          This value is a UTC time value formatted according to ISO 8601 (for example,
          `2021-06-26T12:00:00Z` for noon UTC on 26 June 2021).
    :attr str state: (optional) An encoded string that represents the configuration
          state of the assistant at the beginning of the conversation. If you are using
          the stateless `message` method, save this value and then send it in the context
          of the subsequent message request to avoid disruptions if there are
          configuration changes during the conversation (such as a change to a skill the
          assistant uses).
    :attr bool skip_user_input: (optional) For internal use only.
    """

    def __init__(self,
                 *,
                 timezone: str = None,
                 user_id: str = None,
                 turn_count: int = None,
                 locale: str = None,
                 reference_time: str = None,
                 session_start_time: str = None,
                 state: str = None,
                 skip_user_input: bool = None) -> None:
        """
        Initialize a MessageContextGlobalSystem object.

        :param str timezone: (optional) The user time zone. The assistant uses the
               time zone to correctly resolve relative time references.
        :param str user_id: (optional) A string value that identifies the user who
               is interacting with the assistant. The client must provide a unique
               identifier for each individual end user who accesses the application. For
               user-based plans, this user ID is used to identify unique users for billing
               purposes. This string cannot contain carriage return, newline, or tab
               characters. If no value is specified in the input, **user_id** is
               automatically set to the value of **context.global.session_id**.
               **Note:** This property is the same as the **user_id** property at the root
               of the message body. If **user_id** is specified in both locations in a
               message request, the value specified at the root is used.
        :param int turn_count: (optional) A counter that is automatically
               incremented with each turn of the conversation. A value of 1 indicates that
               this is the the first turn of a new conversation, which can affect the
               behavior of some skills (for example, triggering the start node of a
               dialog).
        :param str locale: (optional) The language code for localization in the
               user input. The specified locale overrides the default for the assistant,
               and is used for interpreting entity values in user input such as date
               values. For example, `04/03/2018` might be interpreted either as April 3 or
               March 4, depending on the locale.
                This property is included only if the new system entities are enabled for
               the skill.
        :param str reference_time: (optional) The base time for interpreting any
               relative time mentions in the user input. The specified time overrides the
               current server time, and is used to calculate times mentioned in relative
               terms such as `now` or `tomorrow`. This can be useful for simulating past
               or future times for testing purposes, or when analyzing documents such as
               news articles.
               This value must be a UTC time value formatted according to ISO 8601 (for
               example, `2021-06-26T12:00:00Z` for noon UTC on 26 June 2021).
               This property is included only if the new system entities are enabled for
               the skill.
        :param str session_start_time: (optional) The time at which the session
               started. With the stateful `message` method, the start time is always
               present, and is set by the service based on the time the session was
               created. With the stateless `message` method, the start time is set by the
               service in the response to the first message, and should be returned as
               part of the context with each subsequent message in the session.
               This value is a UTC time value formatted according to ISO 8601 (for
               example, `2021-06-26T12:00:00Z` for noon UTC on 26 June 2021).
        :param str state: (optional) An encoded string that represents the
               configuration state of the assistant at the beginning of the conversation.
               If you are using the stateless `message` method, save this value and then
               send it in the context of the subsequent message request to avoid
               disruptions if there are configuration changes during the conversation
               (such as a change to a skill the assistant uses).
        :param bool skip_user_input: (optional) For internal use only.
        """
        self.timezone = timezone
        self.user_id = user_id
        self.turn_count = turn_count
        self.locale = locale
        self.reference_time = reference_time
        self.session_start_time = session_start_time
        self.state = state
        self.skip_user_input = skip_user_input

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageContextGlobalSystem':
        """Initialize a MessageContextGlobalSystem object from a json dictionary."""
        args = {}
        if 'timezone' in _dict:
            args['timezone'] = _dict.get('timezone')
        if 'user_id' in _dict:
            args['user_id'] = _dict.get('user_id')
        if 'turn_count' in _dict:
            args['turn_count'] = _dict.get('turn_count')
        if 'locale' in _dict:
            args['locale'] = _dict.get('locale')
        if 'reference_time' in _dict:
            args['reference_time'] = _dict.get('reference_time')
        if 'session_start_time' in _dict:
            args['session_start_time'] = _dict.get('session_start_time')
        if 'state' in _dict:
            args['state'] = _dict.get('state')
        if 'skip_user_input' in _dict:
            args['skip_user_input'] = _dict.get('skip_user_input')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageContextGlobalSystem object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'timezone') and self.timezone is not None:
            _dict['timezone'] = self.timezone
        if hasattr(self, 'user_id') and self.user_id is not None:
            _dict['user_id'] = self.user_id
        if hasattr(self, 'turn_count') and self.turn_count is not None:
            _dict['turn_count'] = self.turn_count
        if hasattr(self, 'locale') and self.locale is not None:
            _dict['locale'] = self.locale
        if hasattr(self, 'reference_time') and self.reference_time is not None:
            _dict['reference_time'] = self.reference_time
        if hasattr(
                self,
                'session_start_time') and self.session_start_time is not None:
            _dict['session_start_time'] = self.session_start_time
        if hasattr(self, 'state') and self.state is not None:
            _dict['state'] = self.state
        if hasattr(self,
                   'skip_user_input') and self.skip_user_input is not None:
            _dict['skip_user_input'] = self.skip_user_input
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageContextGlobalSystem object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageContextGlobalSystem') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageContextGlobalSystem') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class LocaleEnum(str, Enum):
        """
        The language code for localization in the user input. The specified locale
        overrides the default for the assistant, and is used for interpreting entity
        values in user input such as date values. For example, `04/03/2018` might be
        interpreted either as April 3 or March 4, depending on the locale.
         This property is included only if the new system entities are enabled for the
        skill.
        """
        EN_US = 'en-us'
        EN_CA = 'en-ca'
        EN_GB = 'en-gb'
        AR_AR = 'ar-ar'
        CS_CZ = 'cs-cz'
        DE_DE = 'de-de'
        ES_ES = 'es-es'
        FR_FR = 'fr-fr'
        IT_IT = 'it-it'
        JA_JP = 'ja-jp'
        KO_KR = 'ko-kr'
        NL_NL = 'nl-nl'
        PT_BR = 'pt-br'
        ZH_CN = 'zh-cn'
        ZH_TW = 'zh-tw'


class MessageContextSkill():
    """
    Contains information specific to a particular skill used by the assistant. The
    property name must be the same as the name of the skill (for example, `main skill`).

    :attr dict user_defined: (optional) Arbitrary variables that can be read and
          written by a particular skill.
    :attr MessageContextSkillSystem system: (optional) System context data used by
          the skill.
    """

    def __init__(self,
                 *,
                 user_defined: dict = None,
                 system: 'MessageContextSkillSystem' = None) -> None:
        """
        Initialize a MessageContextSkill object.

        :param dict user_defined: (optional) Arbitrary variables that can be read
               and written by a particular skill.
        :param MessageContextSkillSystem system: (optional) System context data
               used by the skill.
        """
        self.user_defined = user_defined
        self.system = system

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageContextSkill':
        """Initialize a MessageContextSkill object from a json dictionary."""
        args = {}
        if 'user_defined' in _dict:
            args['user_defined'] = _dict.get('user_defined')
        if 'system' in _dict:
            args['system'] = MessageContextSkillSystem.from_dict(
                _dict.get('system'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageContextSkill object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'user_defined') and self.user_defined is not None:
            _dict['user_defined'] = self.user_defined
        if hasattr(self, 'system') and self.system is not None:
            _dict['system'] = self.system.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageContextSkill object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageContextSkill') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageContextSkill') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class MessageContextSkillSystem():
    """
    System context data used by the skill.

    :attr str state: (optional) An encoded string that represents the current
          conversation state. By saving this value and then sending it in the context of a
          subsequent message request, you can return to an earlier point in the
          conversation. If you are using stateful sessions, you can also use a stored
          state value to restore a paused conversation whose session is expired.
    """

    # The set of defined properties for the class
    _properties = frozenset(['state'])

    def __init__(self, *, state: str = None, **kwargs) -> None:
        """
        Initialize a MessageContextSkillSystem object.

        :param str state: (optional) An encoded string that represents the current
               conversation state. By saving this value and then sending it in the context
               of a subsequent message request, you can return to an earlier point in the
               conversation. If you are using stateful sessions, you can also use a stored
               state value to restore a paused conversation whose session is expired.
        :param **kwargs: (optional) Any additional properties.
        """
        self.state = state
        for _key, _value in kwargs.items():
            setattr(self, _key, _value)

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageContextSkillSystem':
        """Initialize a MessageContextSkillSystem object from a json dictionary."""
        args = {}
        if 'state' in _dict:
            args['state'] = _dict.get('state')
        args.update(
            {k: v for (k, v) in _dict.items() if k not in cls._properties})
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageContextSkillSystem object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'state') and self.state is not None:
            _dict['state'] = self.state
        for _key in [
                k for k in vars(self).keys()
                if k not in MessageContextSkillSystem._properties
        ]:
            if getattr(self, _key, None) is not None:
                _dict[_key] = getattr(self, _key)
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def get_properties(self) -> Dict:
        """Return a dictionary of arbitrary properties from this instance of MessageContextSkillSystem"""
        _dict = {}

        for _key in [
                k for k in vars(self).keys()
                if k not in MessageContextSkillSystem._properties
        ]:
            _dict[_key] = getattr(self, _key)
        return _dict

    def set_properties(self, _dict: dict):
        """Set a dictionary of arbitrary properties to this instance of MessageContextSkillSystem"""
        for _key in [
                k for k in vars(self).keys()
                if k not in MessageContextSkillSystem._properties
        ]:
            delattr(self, _key)

        for _key, _value in _dict.items():
            if _key not in MessageContextSkillSystem._properties:
                setattr(self, _key, _value)

    def __str__(self) -> str:
        """Return a `str` version of this MessageContextSkillSystem object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageContextSkillSystem') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageContextSkillSystem') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class MessageContextStateless():
    """
    MessageContextStateless.

    :attr MessageContextGlobalStateless global_: (optional) Session context data
          that is shared by all skills used by the assistant.
    :attr dict skills: (optional) Information specific to particular skills used by
          the assistant.
          **Note:** Currently, only a single child property is supported, containing
          variables that apply to the dialog skill used by the assistant.
    :attr object integrations: (optional) An object containing context data that is
          specific to particular integrations. For more information, see the
          [documentation](https://cloud.ibm.com/docs/assistant?topic=assistant-dialog-integrations).
    """

    def __init__(self,
                 *,
                 global_: 'MessageContextGlobalStateless' = None,
                 skills: dict = None,
                 integrations: object = None) -> None:
        """
        Initialize a MessageContextStateless object.

        :param MessageContextGlobalStateless global_: (optional) Session context
               data that is shared by all skills used by the assistant.
        :param dict skills: (optional) Information specific to particular skills
               used by the assistant.
               **Note:** Currently, only a single child property is supported, containing
               variables that apply to the dialog skill used by the assistant.
        :param object integrations: (optional) An object containing context data
               that is specific to particular integrations. For more information, see the
               [documentation](https://cloud.ibm.com/docs/assistant?topic=assistant-dialog-integrations).
        """
        self.global_ = global_
        self.skills = skills
        self.integrations = integrations

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageContextStateless':
        """Initialize a MessageContextStateless object from a json dictionary."""
        args = {}
        if 'global' in _dict:
            args['global_'] = MessageContextGlobalStateless.from_dict(
                _dict.get('global'))
        if 'skills' in _dict:
            args['skills'] = {
                k: MessageContextSkill.from_dict(v)
                for k, v in _dict.get('skills').items()
            }
        if 'integrations' in _dict:
            args['integrations'] = _dict.get('integrations')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageContextStateless object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'global_') and self.global_ is not None:
            _dict['global'] = self.global_.to_dict()
        if hasattr(self, 'skills') and self.skills is not None:
            _dict['skills'] = {k: v.to_dict() for k, v in self.skills.items()}
        if hasattr(self, 'integrations') and self.integrations is not None:
            _dict['integrations'] = self.integrations
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageContextStateless object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageContextStateless') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageContextStateless') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class MessageInput():
    """
    An input object that includes the input text.

    :attr str message_type: (optional) The type of the message:
          - `text`: The user input is processed normally by the assistant.
          - `search`: Only search results are returned. (Any dialog or actions skill is
          bypassed.)
          **Note:** A `search` message results in an error if no search skill is
          configured for the assistant.
    :attr str text: (optional) The text of the user input. This string cannot
          contain carriage return, newline, or tab characters.
    :attr List[RuntimeIntent] intents: (optional) Intents to use when evaluating the
          user input. Include intents from the previous response to continue using those
          intents rather than trying to recognize intents in the new input.
    :attr List[RuntimeEntity] entities: (optional) Entities to use when evaluating
          the message. Include entities from the previous response to continue using those
          entities rather than detecting entities in the new input.
    :attr str suggestion_id: (optional) For internal use only.
    :attr List[MessageInputAttachment] attachments: (optional) An array of
          multimedia attachments to be sent with the message.
          **Note:** Attachments are not processed by the assistant itself, but can be sent
          to external services by webhooks.
    :attr MessageInputOptions options: (optional) Optional properties that control
          how the assistant responds.
    """

    def __init__(self,
                 *,
                 message_type: str = None,
                 text: str = None,
                 intents: List['RuntimeIntent'] = None,
                 entities: List['RuntimeEntity'] = None,
                 suggestion_id: str = None,
                 attachments: List['MessageInputAttachment'] = None,
                 options: 'MessageInputOptions' = None) -> None:
        """
        Initialize a MessageInput object.

        :param str message_type: (optional) The type of the message:
               - `text`: The user input is processed normally by the assistant.
               - `search`: Only search results are returned. (Any dialog or actions skill
               is bypassed.)
               **Note:** A `search` message results in an error if no search skill is
               configured for the assistant.
        :param str text: (optional) The text of the user input. This string cannot
               contain carriage return, newline, or tab characters.
        :param List[RuntimeIntent] intents: (optional) Intents to use when
               evaluating the user input. Include intents from the previous response to
               continue using those intents rather than trying to recognize intents in the
               new input.
        :param List[RuntimeEntity] entities: (optional) Entities to use when
               evaluating the message. Include entities from the previous response to
               continue using those entities rather than detecting entities in the new
               input.
        :param str suggestion_id: (optional) For internal use only.
        :param List[MessageInputAttachment] attachments: (optional) An array of
               multimedia attachments to be sent with the message.
               **Note:** Attachments are not processed by the assistant itself, but can be
               sent to external services by webhooks.
        :param MessageInputOptions options: (optional) Optional properties that
               control how the assistant responds.
        """
        self.message_type = message_type
        self.text = text
        self.intents = intents
        self.entities = entities
        self.suggestion_id = suggestion_id
        self.attachments = attachments
        self.options = options

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageInput':
        """Initialize a MessageInput object from a json dictionary."""
        args = {}
        if 'message_type' in _dict:
            args['message_type'] = _dict.get('message_type')
        if 'text' in _dict:
            args['text'] = _dict.get('text')
        if 'intents' in _dict:
            args['intents'] = [
                RuntimeIntent.from_dict(x) for x in _dict.get('intents')
            ]
        if 'entities' in _dict:
            args['entities'] = [
                RuntimeEntity.from_dict(x) for x in _dict.get('entities')
            ]
        if 'suggestion_id' in _dict:
            args['suggestion_id'] = _dict.get('suggestion_id')
        if 'attachments' in _dict:
            args['attachments'] = [
                MessageInputAttachment.from_dict(x)
                for x in _dict.get('attachments')
            ]
        if 'options' in _dict:
            args['options'] = MessageInputOptions.from_dict(
                _dict.get('options'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageInput object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'message_type') and self.message_type is not None:
            _dict['message_type'] = self.message_type
        if hasattr(self, 'text') and self.text is not None:
            _dict['text'] = self.text
        if hasattr(self, 'intents') and self.intents is not None:
            _dict['intents'] = [x.to_dict() for x in self.intents]
        if hasattr(self, 'entities') and self.entities is not None:
            _dict['entities'] = [x.to_dict() for x in self.entities]
        if hasattr(self, 'suggestion_id') and self.suggestion_id is not None:
            _dict['suggestion_id'] = self.suggestion_id
        if hasattr(self, 'attachments') and self.attachments is not None:
            _dict['attachments'] = [x.to_dict() for x in self.attachments]
        if hasattr(self, 'options') and self.options is not None:
            _dict['options'] = self.options.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageInput object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageInput') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageInput') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class MessageTypeEnum(str, Enum):
        """
        The type of the message:
        - `text`: The user input is processed normally by the assistant.
        - `search`: Only search results are returned. (Any dialog or actions skill is
        bypassed.)
        **Note:** A `search` message results in an error if no search skill is configured
        for the assistant.
        """
        TEXT = 'text'
        SEARCH = 'search'


class MessageInputAttachment():
    """
    A reference to a media file to be sent as an attachment with the message.

    :attr str url: The URL of the media file.
    :attr str media_type: (optional) The media content type (such as a MIME type) of
          the attachment.
    """

    def __init__(self, url: str, *, media_type: str = None) -> None:
        """
        Initialize a MessageInputAttachment object.

        :param str url: The URL of the media file.
        :param str media_type: (optional) The media content type (such as a MIME
               type) of the attachment.
        """
        self.url = url
        self.media_type = media_type

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageInputAttachment':
        """Initialize a MessageInputAttachment object from a json dictionary."""
        args = {}
        if 'url' in _dict:
            args['url'] = _dict.get('url')
        else:
            raise ValueError(
                'Required property \'url\' not present in MessageInputAttachment JSON'
            )
        if 'media_type' in _dict:
            args['media_type'] = _dict.get('media_type')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageInputAttachment object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'url') and self.url is not None:
            _dict['url'] = self.url
        if hasattr(self, 'media_type') and self.media_type is not None:
            _dict['media_type'] = self.media_type
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageInputAttachment object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageInputAttachment') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageInputAttachment') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class MessageInputOptions():
    """
    Optional properties that control how the assistant responds.

    :attr bool restart: (optional) Whether to restart dialog processing at the root
          of the dialog, regardless of any previously visited nodes. **Note:** This does
          not affect `turn_count` or any other context variables.
    :attr bool alternate_intents: (optional) Whether to return more than one intent.
          Set to `true` to return all matching intents.
    :attr MessageInputOptionsSpelling spelling: (optional) Spelling correction
          options for the message. Any options specified on an individual message override
          the settings configured for the skill.
    :attr bool debug: (optional) Whether to return additional diagnostic
          information. Set to `true` to return additional information in the
          `output.debug` property. If you also specify **return_context**=`true`, the
          returned skill context includes the `system.state` property.
    :attr bool return_context: (optional) Whether to return session context with the
          response. If you specify `true`, the response includes the `context` property.
          If you also specify **debug**=`true`, the returned skill context includes the
          `system.state` property.
    :attr bool export: (optional) Whether to return session context, including full
          conversation state. If you specify `true`, the response includes the `context`
          property, and the skill context includes the `system.state` property.
          **Note:** If **export**=`true`, the context is returned regardless of the value
          of **return_context**.
    """

    def __init__(self,
                 *,
                 restart: bool = None,
                 alternate_intents: bool = None,
                 spelling: 'MessageInputOptionsSpelling' = None,
                 debug: bool = None,
                 return_context: bool = None,
                 export: bool = None) -> None:
        """
        Initialize a MessageInputOptions object.

        :param bool restart: (optional) Whether to restart dialog processing at the
               root of the dialog, regardless of any previously visited nodes. **Note:**
               This does not affect `turn_count` or any other context variables.
        :param bool alternate_intents: (optional) Whether to return more than one
               intent. Set to `true` to return all matching intents.
        :param MessageInputOptionsSpelling spelling: (optional) Spelling correction
               options for the message. Any options specified on an individual message
               override the settings configured for the skill.
        :param bool debug: (optional) Whether to return additional diagnostic
               information. Set to `true` to return additional information in the
               `output.debug` property. If you also specify **return_context**=`true`, the
               returned skill context includes the `system.state` property.
        :param bool return_context: (optional) Whether to return session context
               with the response. If you specify `true`, the response includes the
               `context` property. If you also specify **debug**=`true`, the returned
               skill context includes the `system.state` property.
        :param bool export: (optional) Whether to return session context, including
               full conversation state. If you specify `true`, the response includes the
               `context` property, and the skill context includes the `system.state`
               property.
               **Note:** If **export**=`true`, the context is returned regardless of the
               value of **return_context**.
        """
        self.restart = restart
        self.alternate_intents = alternate_intents
        self.spelling = spelling
        self.debug = debug
        self.return_context = return_context
        self.export = export

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageInputOptions':
        """Initialize a MessageInputOptions object from a json dictionary."""
        args = {}
        if 'restart' in _dict:
            args['restart'] = _dict.get('restart')
        if 'alternate_intents' in _dict:
            args['alternate_intents'] = _dict.get('alternate_intents')
        if 'spelling' in _dict:
            args['spelling'] = MessageInputOptionsSpelling.from_dict(
                _dict.get('spelling'))
        if 'debug' in _dict:
            args['debug'] = _dict.get('debug')
        if 'return_context' in _dict:
            args['return_context'] = _dict.get('return_context')
        if 'export' in _dict:
            args['export'] = _dict.get('export')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageInputOptions object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'restart') and self.restart is not None:
            _dict['restart'] = self.restart
        if hasattr(self,
                   'alternate_intents') and self.alternate_intents is not None:
            _dict['alternate_intents'] = self.alternate_intents
        if hasattr(self, 'spelling') and self.spelling is not None:
            _dict['spelling'] = self.spelling.to_dict()
        if hasattr(self, 'debug') and self.debug is not None:
            _dict['debug'] = self.debug
        if hasattr(self, 'return_context') and self.return_context is not None:
            _dict['return_context'] = self.return_context
        if hasattr(self, 'export') and self.export is not None:
            _dict['export'] = self.export
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageInputOptions object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageInputOptions') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageInputOptions') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class MessageInputOptionsSpelling():
    """
    Spelling correction options for the message. Any options specified on an individual
    message override the settings configured for the skill.

    :attr bool suggestions: (optional) Whether to use spelling correction when
          processing the input. If spelling correction is used and **auto_correct** is
          `true`, any spelling corrections are automatically applied to the user input. If
          **auto_correct** is `false`, any suggested corrections are returned in the
          **output.spelling** property.
          This property overrides the value of the **spelling_suggestions** property in
          the workspace settings for the skill.
    :attr bool auto_correct: (optional) Whether to use autocorrection when
          processing the input. If this property is `true`, any corrections are
          automatically applied to the user input, and the original text is returned in
          the **output.spelling** property of the message response. This property
          overrides the value of the **spelling_auto_correct** property in the workspace
          settings for the skill.
    """

    def __init__(self,
                 *,
                 suggestions: bool = None,
                 auto_correct: bool = None) -> None:
        """
        Initialize a MessageInputOptionsSpelling object.

        :param bool suggestions: (optional) Whether to use spelling correction when
               processing the input. If spelling correction is used and **auto_correct**
               is `true`, any spelling corrections are automatically applied to the user
               input. If **auto_correct** is `false`, any suggested corrections are
               returned in the **output.spelling** property.
               This property overrides the value of the **spelling_suggestions** property
               in the workspace settings for the skill.
        :param bool auto_correct: (optional) Whether to use autocorrection when
               processing the input. If this property is `true`, any corrections are
               automatically applied to the user input, and the original text is returned
               in the **output.spelling** property of the message response. This property
               overrides the value of the **spelling_auto_correct** property in the
               workspace settings for the skill.
        """
        self.suggestions = suggestions
        self.auto_correct = auto_correct

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageInputOptionsSpelling':
        """Initialize a MessageInputOptionsSpelling object from a json dictionary."""
        args = {}
        if 'suggestions' in _dict:
            args['suggestions'] = _dict.get('suggestions')
        if 'auto_correct' in _dict:
            args['auto_correct'] = _dict.get('auto_correct')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageInputOptionsSpelling object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'suggestions') and self.suggestions is not None:
            _dict['suggestions'] = self.suggestions
        if hasattr(self, 'auto_correct') and self.auto_correct is not None:
            _dict['auto_correct'] = self.auto_correct
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageInputOptionsSpelling object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageInputOptionsSpelling') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageInputOptionsSpelling') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class MessageInputOptionsStateless():
    """
    Optional properties that control how the assistant responds.

    :attr bool restart: (optional) Whether to restart dialog processing at the root
          of the dialog, regardless of any previously visited nodes. **Note:** This does
          not affect `turn_count` or any other context variables.
    :attr bool alternate_intents: (optional) Whether to return more than one intent.
          Set to `true` to return all matching intents.
    :attr MessageInputOptionsSpelling spelling: (optional) Spelling correction
          options for the message. Any options specified on an individual message override
          the settings configured for the skill.
    :attr bool debug: (optional) Whether to return additional diagnostic
          information. Set to `true` to return additional information in the
          `output.debug` property.
    """

    def __init__(self,
                 *,
                 restart: bool = None,
                 alternate_intents: bool = None,
                 spelling: 'MessageInputOptionsSpelling' = None,
                 debug: bool = None) -> None:
        """
        Initialize a MessageInputOptionsStateless object.

        :param bool restart: (optional) Whether to restart dialog processing at the
               root of the dialog, regardless of any previously visited nodes. **Note:**
               This does not affect `turn_count` or any other context variables.
        :param bool alternate_intents: (optional) Whether to return more than one
               intent. Set to `true` to return all matching intents.
        :param MessageInputOptionsSpelling spelling: (optional) Spelling correction
               options for the message. Any options specified on an individual message
               override the settings configured for the skill.
        :param bool debug: (optional) Whether to return additional diagnostic
               information. Set to `true` to return additional information in the
               `output.debug` property.
        """
        self.restart = restart
        self.alternate_intents = alternate_intents
        self.spelling = spelling
        self.debug = debug

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageInputOptionsStateless':
        """Initialize a MessageInputOptionsStateless object from a json dictionary."""
        args = {}
        if 'restart' in _dict:
            args['restart'] = _dict.get('restart')
        if 'alternate_intents' in _dict:
            args['alternate_intents'] = _dict.get('alternate_intents')
        if 'spelling' in _dict:
            args['spelling'] = MessageInputOptionsSpelling.from_dict(
                _dict.get('spelling'))
        if 'debug' in _dict:
            args['debug'] = _dict.get('debug')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageInputOptionsStateless object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'restart') and self.restart is not None:
            _dict['restart'] = self.restart
        if hasattr(self,
                   'alternate_intents') and self.alternate_intents is not None:
            _dict['alternate_intents'] = self.alternate_intents
        if hasattr(self, 'spelling') and self.spelling is not None:
            _dict['spelling'] = self.spelling.to_dict()
        if hasattr(self, 'debug') and self.debug is not None:
            _dict['debug'] = self.debug
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageInputOptionsStateless object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageInputOptionsStateless') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageInputOptionsStateless') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class MessageInputStateless():
    """
    An input object that includes the input text.

    :attr str message_type: (optional) The type of the message:
          - `text`: The user input is processed normally by the assistant.
          - `search`: Only search results are returned. (Any dialog or actions skill is
          bypassed.)
          **Note:** A `search` message results in an error if no search skill is
          configured for the assistant.
    :attr str text: (optional) The text of the user input. This string cannot
          contain carriage return, newline, or tab characters.
    :attr List[RuntimeIntent] intents: (optional) Intents to use when evaluating the
          user input. Include intents from the previous response to continue using those
          intents rather than trying to recognize intents in the new input.
    :attr List[RuntimeEntity] entities: (optional) Entities to use when evaluating
          the message. Include entities from the previous response to continue using those
          entities rather than detecting entities in the new input.
    :attr str suggestion_id: (optional) For internal use only.
    :attr List[MessageInputAttachment] attachments: (optional) An array of
          multimedia attachments to be sent with the message.
          **Note:** Attachments are not processed by the assistant itself, but can be sent
          to external services by webhooks.
    :attr MessageInputOptionsStateless options: (optional) Optional properties that
          control how the assistant responds.
    """

    def __init__(self,
                 *,
                 message_type: str = None,
                 text: str = None,
                 intents: List['RuntimeIntent'] = None,
                 entities: List['RuntimeEntity'] = None,
                 suggestion_id: str = None,
                 attachments: List['MessageInputAttachment'] = None,
                 options: 'MessageInputOptionsStateless' = None) -> None:
        """
        Initialize a MessageInputStateless object.

        :param str message_type: (optional) The type of the message:
               - `text`: The user input is processed normally by the assistant.
               - `search`: Only search results are returned. (Any dialog or actions skill
               is bypassed.)
               **Note:** A `search` message results in an error if no search skill is
               configured for the assistant.
        :param str text: (optional) The text of the user input. This string cannot
               contain carriage return, newline, or tab characters.
        :param List[RuntimeIntent] intents: (optional) Intents to use when
               evaluating the user input. Include intents from the previous response to
               continue using those intents rather than trying to recognize intents in the
               new input.
        :param List[RuntimeEntity] entities: (optional) Entities to use when
               evaluating the message. Include entities from the previous response to
               continue using those entities rather than detecting entities in the new
               input.
        :param str suggestion_id: (optional) For internal use only.
        :param List[MessageInputAttachment] attachments: (optional) An array of
               multimedia attachments to be sent with the message.
               **Note:** Attachments are not processed by the assistant itself, but can be
               sent to external services by webhooks.
        :param MessageInputOptionsStateless options: (optional) Optional properties
               that control how the assistant responds.
        """
        self.message_type = message_type
        self.text = text
        self.intents = intents
        self.entities = entities
        self.suggestion_id = suggestion_id
        self.attachments = attachments
        self.options = options

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageInputStateless':
        """Initialize a MessageInputStateless object from a json dictionary."""
        args = {}
        if 'message_type' in _dict:
            args['message_type'] = _dict.get('message_type')
        if 'text' in _dict:
            args['text'] = _dict.get('text')
        if 'intents' in _dict:
            args['intents'] = [
                RuntimeIntent.from_dict(x) for x in _dict.get('intents')
            ]
        if 'entities' in _dict:
            args['entities'] = [
                RuntimeEntity.from_dict(x) for x in _dict.get('entities')
            ]
        if 'suggestion_id' in _dict:
            args['suggestion_id'] = _dict.get('suggestion_id')
        if 'attachments' in _dict:
            args['attachments'] = [
                MessageInputAttachment.from_dict(x)
                for x in _dict.get('attachments')
            ]
        if 'options' in _dict:
            args['options'] = MessageInputOptionsStateless.from_dict(
                _dict.get('options'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageInputStateless object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'message_type') and self.message_type is not None:
            _dict['message_type'] = self.message_type
        if hasattr(self, 'text') and self.text is not None:
            _dict['text'] = self.text
        if hasattr(self, 'intents') and self.intents is not None:
            _dict['intents'] = [x.to_dict() for x in self.intents]
        if hasattr(self, 'entities') and self.entities is not None:
            _dict['entities'] = [x.to_dict() for x in self.entities]
        if hasattr(self, 'suggestion_id') and self.suggestion_id is not None:
            _dict['suggestion_id'] = self.suggestion_id
        if hasattr(self, 'attachments') and self.attachments is not None:
            _dict['attachments'] = [x.to_dict() for x in self.attachments]
        if hasattr(self, 'options') and self.options is not None:
            _dict['options'] = self.options.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageInputStateless object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageInputStateless') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageInputStateless') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class MessageTypeEnum(str, Enum):
        """
        The type of the message:
        - `text`: The user input is processed normally by the assistant.
        - `search`: Only search results are returned. (Any dialog or actions skill is
        bypassed.)
        **Note:** A `search` message results in an error if no search skill is configured
        for the assistant.
        """
        TEXT = 'text'
        SEARCH = 'search'


class MessageOutput():
    """
    Assistant output to be rendered or processed by the client.

    :attr List[RuntimeResponseGeneric] generic: (optional) Output intended for any
          channel. It is the responsibility of the client application to implement the
          supported response types.
    :attr List[RuntimeIntent] intents: (optional) An array of intents recognized in
          the user input, sorted in descending order of confidence.
    :attr List[RuntimeEntity] entities: (optional) An array of entities identified
          in the user input.
    :attr List[DialogNodeAction] actions: (optional) An array of objects describing
          any actions requested by the dialog node.
    :attr MessageOutputDebug debug: (optional) Additional detailed information about
          a message response and how it was generated.
    :attr dict user_defined: (optional) An object containing any custom properties
          included in the response. This object includes any arbitrary properties defined
          in the dialog JSON editor as part of the dialog node output.
    :attr MessageOutputSpelling spelling: (optional) Properties describing any
          spelling corrections in the user input that was received.
    """

    def __init__(self,
                 *,
                 generic: List['RuntimeResponseGeneric'] = None,
                 intents: List['RuntimeIntent'] = None,
                 entities: List['RuntimeEntity'] = None,
                 actions: List['DialogNodeAction'] = None,
                 debug: 'MessageOutputDebug' = None,
                 user_defined: dict = None,
                 spelling: 'MessageOutputSpelling' = None) -> None:
        """
        Initialize a MessageOutput object.

        :param List[RuntimeResponseGeneric] generic: (optional) Output intended for
               any channel. It is the responsibility of the client application to
               implement the supported response types.
        :param List[RuntimeIntent] intents: (optional) An array of intents
               recognized in the user input, sorted in descending order of confidence.
        :param List[RuntimeEntity] entities: (optional) An array of entities
               identified in the user input.
        :param List[DialogNodeAction] actions: (optional) An array of objects
               describing any actions requested by the dialog node.
        :param MessageOutputDebug debug: (optional) Additional detailed information
               about a message response and how it was generated.
        :param dict user_defined: (optional) An object containing any custom
               properties included in the response. This object includes any arbitrary
               properties defined in the dialog JSON editor as part of the dialog node
               output.
        :param MessageOutputSpelling spelling: (optional) Properties describing any
               spelling corrections in the user input that was received.
        """
        self.generic = generic
        self.intents = intents
        self.entities = entities
        self.actions = actions
        self.debug = debug
        self.user_defined = user_defined
        self.spelling = spelling

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageOutput':
        """Initialize a MessageOutput object from a json dictionary."""
        args = {}
        if 'generic' in _dict:
            args['generic'] = [
                RuntimeResponseGeneric.from_dict(x)
                for x in _dict.get('generic')
            ]
        if 'intents' in _dict:
            args['intents'] = [
                RuntimeIntent.from_dict(x) for x in _dict.get('intents')
            ]
        if 'entities' in _dict:
            args['entities'] = [
                RuntimeEntity.from_dict(x) for x in _dict.get('entities')
            ]
        if 'actions' in _dict:
            args['actions'] = [
                DialogNodeAction.from_dict(x) for x in _dict.get('actions')
            ]
        if 'debug' in _dict:
            args['debug'] = MessageOutputDebug.from_dict(_dict.get('debug'))
        if 'user_defined' in _dict:
            args['user_defined'] = _dict.get('user_defined')
        if 'spelling' in _dict:
            args['spelling'] = MessageOutputSpelling.from_dict(
                _dict.get('spelling'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageOutput object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'generic') and self.generic is not None:
            _dict['generic'] = [x.to_dict() for x in self.generic]
        if hasattr(self, 'intents') and self.intents is not None:
            _dict['intents'] = [x.to_dict() for x in self.intents]
        if hasattr(self, 'entities') and self.entities is not None:
            _dict['entities'] = [x.to_dict() for x in self.entities]
        if hasattr(self, 'actions') and self.actions is not None:
            _dict['actions'] = [x.to_dict() for x in self.actions]
        if hasattr(self, 'debug') and self.debug is not None:
            _dict['debug'] = self.debug.to_dict()
        if hasattr(self, 'user_defined') and self.user_defined is not None:
            _dict['user_defined'] = self.user_defined
        if hasattr(self, 'spelling') and self.spelling is not None:
            _dict['spelling'] = self.spelling.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageOutput object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageOutput') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageOutput') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class MessageOutputDebug():
    """
    Additional detailed information about a message response and how it was generated.

    :attr List[DialogNodeVisited] nodes_visited: (optional) An array of objects
          containing detailed diagnostic information about dialog nodes that were
          triggered during processing of the input message.
    :attr List[DialogLogMessage] log_messages: (optional) An array of up to 50
          messages logged with the request.
    :attr bool branch_exited: (optional) Assistant sets this to true when this
          message response concludes or interrupts a dialog.
    :attr str branch_exited_reason: (optional) When `branch_exited` is set to `true`
          by the assistant, the `branch_exited_reason` specifies whether the dialog
          completed by itself or got interrupted.
    """

    def __init__(self,
                 *,
                 nodes_visited: List['DialogNodeVisited'] = None,
                 log_messages: List['DialogLogMessage'] = None,
                 branch_exited: bool = None,
                 branch_exited_reason: str = None) -> None:
        """
        Initialize a MessageOutputDebug object.

        :param List[DialogNodeVisited] nodes_visited: (optional) An array of
               objects containing detailed diagnostic information about dialog nodes that
               were triggered during processing of the input message.
        :param List[DialogLogMessage] log_messages: (optional) An array of up to 50
               messages logged with the request.
        :param bool branch_exited: (optional) Assistant sets this to true when this
               message response concludes or interrupts a dialog.
        :param str branch_exited_reason: (optional) When `branch_exited` is set to
               `true` by the assistant, the `branch_exited_reason` specifies whether the
               dialog completed by itself or got interrupted.
        """
        self.nodes_visited = nodes_visited
        self.log_messages = log_messages
        self.branch_exited = branch_exited
        self.branch_exited_reason = branch_exited_reason

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageOutputDebug':
        """Initialize a MessageOutputDebug object from a json dictionary."""
        args = {}
        if 'nodes_visited' in _dict:
            args['nodes_visited'] = [
                DialogNodeVisited.from_dict(x)
                for x in _dict.get('nodes_visited')
            ]
        if 'log_messages' in _dict:
            args['log_messages'] = [
                DialogLogMessage.from_dict(x) for x in _dict.get('log_messages')
            ]
        if 'branch_exited' in _dict:
            args['branch_exited'] = _dict.get('branch_exited')
        if 'branch_exited_reason' in _dict:
            args['branch_exited_reason'] = _dict.get('branch_exited_reason')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageOutputDebug object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'nodes_visited') and self.nodes_visited is not None:
            _dict['nodes_visited'] = [x.to_dict() for x in self.nodes_visited]
        if hasattr(self, 'log_messages') and self.log_messages is not None:
            _dict['log_messages'] = [x.to_dict() for x in self.log_messages]
        if hasattr(self, 'branch_exited') and self.branch_exited is not None:
            _dict['branch_exited'] = self.branch_exited
        if hasattr(self, 'branch_exited_reason'
                  ) and self.branch_exited_reason is not None:
            _dict['branch_exited_reason'] = self.branch_exited_reason
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageOutputDebug object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageOutputDebug') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageOutputDebug') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class BranchExitedReasonEnum(str, Enum):
        """
        When `branch_exited` is set to `true` by the assistant, the `branch_exited_reason`
        specifies whether the dialog completed by itself or got interrupted.
        """
        COMPLETED = 'completed'
        FALLBACK = 'fallback'


class MessageOutputSpelling():
    """
    Properties describing any spelling corrections in the user input that was received.

    :attr str text: (optional) The user input text that was used to generate the
          response. If spelling autocorrection is enabled, this text reflects any spelling
          corrections that were applied.
    :attr str original_text: (optional) The original user input text. This property
          is returned only if autocorrection is enabled and the user input was corrected.
    :attr str suggested_text: (optional) Any suggested corrections of the input
          text. This property is returned only if spelling correction is enabled and
          autocorrection is disabled.
    """

    def __init__(self,
                 *,
                 text: str = None,
                 original_text: str = None,
                 suggested_text: str = None) -> None:
        """
        Initialize a MessageOutputSpelling object.

        :param str text: (optional) The user input text that was used to generate
               the response. If spelling autocorrection is enabled, this text reflects any
               spelling corrections that were applied.
        :param str original_text: (optional) The original user input text. This
               property is returned only if autocorrection is enabled and the user input
               was corrected.
        :param str suggested_text: (optional) Any suggested corrections of the
               input text. This property is returned only if spelling correction is
               enabled and autocorrection is disabled.
        """
        self.text = text
        self.original_text = original_text
        self.suggested_text = suggested_text

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageOutputSpelling':
        """Initialize a MessageOutputSpelling object from a json dictionary."""
        args = {}
        if 'text' in _dict:
            args['text'] = _dict.get('text')
        if 'original_text' in _dict:
            args['original_text'] = _dict.get('original_text')
        if 'suggested_text' in _dict:
            args['suggested_text'] = _dict.get('suggested_text')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageOutputSpelling object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'text') and self.text is not None:
            _dict['text'] = self.text
        if hasattr(self, 'original_text') and self.original_text is not None:
            _dict['original_text'] = self.original_text
        if hasattr(self, 'suggested_text') and self.suggested_text is not None:
            _dict['suggested_text'] = self.suggested_text
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageOutputSpelling object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageOutputSpelling') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageOutputSpelling') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class MessageRequest():
    """
    A stateful message request formatted for the Watson Assistant service.

    :attr MessageInput input: (optional) An input object that includes the input
          text.
    :attr MessageContext context: (optional) Context data for the conversation. You
          can use this property to set or modify context variables, which can also be
          accessed by dialog nodes. The context is stored by the assistant on a
          per-session basis.
          **Note:** The total size of the context data stored for a stateful session
          cannot exceed 100KB.
    :attr str user_id: (optional) A string value that identifies the user who is
          interacting with the assistant. The client must provide a unique identifier for
          each individual end user who accesses the application. For user-based plans,
          this user ID is used to identify unique users for billing purposes. This string
          cannot contain carriage return, newline, or tab characters. If no value is
          specified in the input, **user_id** is automatically set to the value of
          **context.global.session_id**.
          **Note:** This property is the same as the **user_id** property in the global
          system context. If **user_id** is specified in both locations, the value
          specified at the root is used.
    """

    def __init__(self,
                 *,
                 input: 'MessageInput' = None,
                 context: 'MessageContext' = None,
                 user_id: str = None) -> None:
        """
        Initialize a MessageRequest object.

        :param MessageInput input: (optional) An input object that includes the
               input text.
        :param MessageContext context: (optional) Context data for the
               conversation. You can use this property to set or modify context variables,
               which can also be accessed by dialog nodes. The context is stored by the
               assistant on a per-session basis.
               **Note:** The total size of the context data stored for a stateful session
               cannot exceed 100KB.
        :param str user_id: (optional) A string value that identifies the user who
               is interacting with the assistant. The client must provide a unique
               identifier for each individual end user who accesses the application. For
               user-based plans, this user ID is used to identify unique users for billing
               purposes. This string cannot contain carriage return, newline, or tab
               characters. If no value is specified in the input, **user_id** is
               automatically set to the value of **context.global.session_id**.
               **Note:** This property is the same as the **user_id** property in the
               global system context. If **user_id** is specified in both locations, the
               value specified at the root is used.
        """
        self.input = input
        self.context = context
        self.user_id = user_id

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageRequest':
        """Initialize a MessageRequest object from a json dictionary."""
        args = {}
        if 'input' in _dict:
            args['input'] = MessageInput.from_dict(_dict.get('input'))
        if 'context' in _dict:
            args['context'] = MessageContext.from_dict(_dict.get('context'))
        if 'user_id' in _dict:
            args['user_id'] = _dict.get('user_id')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageRequest object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'input') and self.input is not None:
            _dict['input'] = self.input.to_dict()
        if hasattr(self, 'context') and self.context is not None:
            _dict['context'] = self.context.to_dict()
        if hasattr(self, 'user_id') and self.user_id is not None:
            _dict['user_id'] = self.user_id
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageRequest object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageRequest') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageRequest') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class MessageResponse():
    """
    A response from the Watson Assistant service.

    :attr MessageOutput output: Assistant output to be rendered or processed by the
          client.
    :attr MessageContext context: (optional) Context data for the conversation. You
          can use this property to access context variables. The context is stored by the
          assistant on a per-session basis.
          **Note:** The context is included in message responses only if
          **return_context**=`true` in the message request. Full context is always
          included in logs.
    :attr str user_id: A string value that identifies the user who is interacting
          with the assistant. The client must provide a unique identifier for each
          individual end user who accesses the application. For user-based plans, this
          user ID is used to identify unique users for billing purposes. This string
          cannot contain carriage return, newline, or tab characters. If no value is
          specified in the input, **user_id** is automatically set to the value of
          **context.global.session_id**.
          **Note:** This property is the same as the **user_id** property in the global
          system context.
    """

    def __init__(self,
                 output: 'MessageOutput',
                 user_id: str,
                 *,
                 context: 'MessageContext' = None) -> None:
        """
        Initialize a MessageResponse object.

        :param MessageOutput output: Assistant output to be rendered or processed
               by the client.
        :param str user_id: A string value that identifies the user who is
               interacting with the assistant. The client must provide a unique identifier
               for each individual end user who accesses the application. For user-based
               plans, this user ID is used to identify unique users for billing purposes.
               This string cannot contain carriage return, newline, or tab characters. If
               no value is specified in the input, **user_id** is automatically set to the
               value of **context.global.session_id**.
               **Note:** This property is the same as the **user_id** property in the
               global system context.
        :param MessageContext context: (optional) Context data for the
               conversation. You can use this property to access context variables. The
               context is stored by the assistant on a per-session basis.
               **Note:** The context is included in message responses only if
               **return_context**=`true` in the message request. Full context is always
               included in logs.
        """
        self.output = output
        self.context = context
        self.user_id = user_id

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageResponse':
        """Initialize a MessageResponse object from a json dictionary."""
        args = {}
        if 'output' in _dict:
            args['output'] = MessageOutput.from_dict(_dict.get('output'))
        else:
            raise ValueError(
                'Required property \'output\' not present in MessageResponse JSON'
            )
        if 'context' in _dict:
            args['context'] = MessageContext.from_dict(_dict.get('context'))
        if 'user_id' in _dict:
            args['user_id'] = _dict.get('user_id')
        else:
            raise ValueError(
                'Required property \'user_id\' not present in MessageResponse JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageResponse object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'output') and self.output is not None:
            _dict['output'] = self.output.to_dict()
        if hasattr(self, 'context') and self.context is not None:
            _dict['context'] = self.context.to_dict()
        if hasattr(self, 'user_id') and self.user_id is not None:
            _dict['user_id'] = self.user_id
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageResponse object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageResponse') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageResponse') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class MessageResponseStateless():
    """
    A stateless response from the Watson Assistant service.

    :attr MessageOutput output: Assistant output to be rendered or processed by the
          client.
    :attr MessageContextStateless context: Context data for the conversation. You
          can use this property to access context variables. The context is not stored by
          the assistant; to maintain session state, include the context from the response
          in the next message.
    :attr str user_id: (optional) A string value that identifies the user who is
          interacting with the assistant. The client must provide a unique identifier for
          each individual end user who accesses the application. For user-based plans,
          this user ID is used to identify unique users for billing purposes. This string
          cannot contain carriage return, newline, or tab characters. If no value is
          specified in the input, **user_id** is automatically set to the value of
          **context.global.session_id**.
          **Note:** This property is the same as the **user_id** property in the global
          system context.
    """

    def __init__(self,
                 output: 'MessageOutput',
                 context: 'MessageContextStateless',
                 *,
                 user_id: str = None) -> None:
        """
        Initialize a MessageResponseStateless object.

        :param MessageOutput output: Assistant output to be rendered or processed
               by the client.
        :param MessageContextStateless context: Context data for the conversation.
               You can use this property to access context variables. The context is not
               stored by the assistant; to maintain session state, include the context
               from the response in the next message.
        :param str user_id: (optional) A string value that identifies the user who
               is interacting with the assistant. The client must provide a unique
               identifier for each individual end user who accesses the application. For
               user-based plans, this user ID is used to identify unique users for billing
               purposes. This string cannot contain carriage return, newline, or tab
               characters. If no value is specified in the input, **user_id** is
               automatically set to the value of **context.global.session_id**.
               **Note:** This property is the same as the **user_id** property in the
               global system context.
        """
        self.output = output
        self.context = context
        self.user_id = user_id

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'MessageResponseStateless':
        """Initialize a MessageResponseStateless object from a json dictionary."""
        args = {}
        if 'output' in _dict:
            args['output'] = MessageOutput.from_dict(_dict.get('output'))
        else:
            raise ValueError(
                'Required property \'output\' not present in MessageResponseStateless JSON'
            )
        if 'context' in _dict:
            args['context'] = MessageContextStateless.from_dict(
                _dict.get('context'))
        else:
            raise ValueError(
                'Required property \'context\' not present in MessageResponseStateless JSON'
            )
        if 'user_id' in _dict:
            args['user_id'] = _dict.get('user_id')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a MessageResponseStateless object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'output') and self.output is not None:
            _dict['output'] = self.output.to_dict()
        if hasattr(self, 'context') and self.context is not None:
            _dict['context'] = self.context.to_dict()
        if hasattr(self, 'user_id') and self.user_id is not None:
            _dict['user_id'] = self.user_id
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this MessageResponseStateless object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'MessageResponseStateless') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'MessageResponseStateless') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class ResponseGenericChannel():
    """
    ResponseGenericChannel.

    :attr str channel: (optional) A channel for which the response is intended.
    """

    def __init__(self, *, channel: str = None) -> None:
        """
        Initialize a ResponseGenericChannel object.

        :param str channel: (optional) A channel for which the response is
               intended.
        """
        self.channel = channel

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'ResponseGenericChannel':
        """Initialize a ResponseGenericChannel object from a json dictionary."""
        args = {}
        if 'channel' in _dict:
            args['channel'] = _dict.get('channel')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a ResponseGenericChannel object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'channel') and self.channel is not None:
            _dict['channel'] = self.channel
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this ResponseGenericChannel object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'ResponseGenericChannel') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'ResponseGenericChannel') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RuntimeEntity():
    """
    The entity value that was recognized in the user input.

    :attr str entity: An entity detected in the input.
    :attr List[int] location: (optional) An array of zero-based character offsets
          that indicate where the detected entity values begin and end in the input text.
    :attr str value: The term in the input text that was recognized as an entity
          value.
    :attr float confidence: (optional) A decimal percentage that represents Watson's
          confidence in the recognized entity.
    :attr List[CaptureGroup] groups: (optional) The recognized capture groups for
          the entity, as defined by the entity pattern.
    :attr RuntimeEntityInterpretation interpretation: (optional) An object
          containing detailed information about the entity recognized in the user input.
          This property is included only if the new system entities are enabled for the
          skill.
          For more information about how the new system entities are interpreted, see the
          [documentation](https://cloud.ibm.com/docs/assistant?topic=assistant-beta-system-entities).
    :attr List[RuntimeEntityAlternative] alternatives: (optional) An array of
          possible alternative values that the user might have intended instead of the
          value returned in the **value** property. This property is returned only for
          `@sys-time` and `@sys-date` entities when the user's input is ambiguous.
          This property is included only if the new system entities are enabled for the
          skill.
    :attr RuntimeEntityRole role: (optional) An object describing the role played by
          a system entity that is specifies the beginning or end of a range recognized in
          the user input. This property is included only if the new system entities are
          enabled for the skill.
    """

    def __init__(self,
                 entity: str,
                 value: str,
                 *,
                 location: List[int] = None,
                 confidence: float = None,
                 groups: List['CaptureGroup'] = None,
                 interpretation: 'RuntimeEntityInterpretation' = None,
                 alternatives: List['RuntimeEntityAlternative'] = None,
                 role: 'RuntimeEntityRole' = None) -> None:
        """
        Initialize a RuntimeEntity object.

        :param str entity: An entity detected in the input.
        :param str value: The term in the input text that was recognized as an
               entity value.
        :param List[int] location: (optional) An array of zero-based character
               offsets that indicate where the detected entity values begin and end in the
               input text.
        :param float confidence: (optional) A decimal percentage that represents
               Watson's confidence in the recognized entity.
        :param List[CaptureGroup] groups: (optional) The recognized capture groups
               for the entity, as defined by the entity pattern.
        :param RuntimeEntityInterpretation interpretation: (optional) An object
               containing detailed information about the entity recognized in the user
               input. This property is included only if the new system entities are
               enabled for the skill.
               For more information about how the new system entities are interpreted, see
               the
               [documentation](https://cloud.ibm.com/docs/assistant?topic=assistant-beta-system-entities).
        :param List[RuntimeEntityAlternative] alternatives: (optional) An array of
               possible alternative values that the user might have intended instead of
               the value returned in the **value** property. This property is returned
               only for `@sys-time` and `@sys-date` entities when the user's input is
               ambiguous.
               This property is included only if the new system entities are enabled for
               the skill.
        :param RuntimeEntityRole role: (optional) An object describing the role
               played by a system entity that is specifies the beginning or end of a range
               recognized in the user input. This property is included only if the new
               system entities are enabled for the skill.
        """
        self.entity = entity
        self.location = location
        self.value = value
        self.confidence = confidence
        self.groups = groups
        self.interpretation = interpretation
        self.alternatives = alternatives
        self.role = role

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'RuntimeEntity':
        """Initialize a RuntimeEntity object from a json dictionary."""
        args = {}
        if 'entity' in _dict:
            args['entity'] = _dict.get('entity')
        else:
            raise ValueError(
                'Required property \'entity\' not present in RuntimeEntity JSON'
            )
        if 'location' in _dict:
            args['location'] = _dict.get('location')
        if 'value' in _dict:
            args['value'] = _dict.get('value')
        else:
            raise ValueError(
                'Required property \'value\' not present in RuntimeEntity JSON')
        if 'confidence' in _dict:
            args['confidence'] = _dict.get('confidence')
        if 'groups' in _dict:
            args['groups'] = [
                CaptureGroup.from_dict(x) for x in _dict.get('groups')
            ]
        if 'interpretation' in _dict:
            args['interpretation'] = RuntimeEntityInterpretation.from_dict(
                _dict.get('interpretation'))
        if 'alternatives' in _dict:
            args['alternatives'] = [
                RuntimeEntityAlternative.from_dict(x)
                for x in _dict.get('alternatives')
            ]
        if 'role' in _dict:
            args['role'] = RuntimeEntityRole.from_dict(_dict.get('role'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeEntity object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'entity') and self.entity is not None:
            _dict['entity'] = self.entity
        if hasattr(self, 'location') and self.location is not None:
            _dict['location'] = self.location
        if hasattr(self, 'value') and self.value is not None:
            _dict['value'] = self.value
        if hasattr(self, 'confidence') and self.confidence is not None:
            _dict['confidence'] = self.confidence
        if hasattr(self, 'groups') and self.groups is not None:
            _dict['groups'] = [x.to_dict() for x in self.groups]
        if hasattr(self, 'interpretation') and self.interpretation is not None:
            _dict['interpretation'] = self.interpretation.to_dict()
        if hasattr(self, 'alternatives') and self.alternatives is not None:
            _dict['alternatives'] = [x.to_dict() for x in self.alternatives]
        if hasattr(self, 'role') and self.role is not None:
            _dict['role'] = self.role.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeEntity object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'RuntimeEntity') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'RuntimeEntity') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RuntimeEntityAlternative():
    """
    An alternative value for the recognized entity.

    :attr str value: (optional) The entity value that was recognized in the user
          input.
    :attr float confidence: (optional) A decimal percentage that represents Watson's
          confidence in the recognized entity.
    """

    def __init__(self, *, value: str = None, confidence: float = None) -> None:
        """
        Initialize a RuntimeEntityAlternative object.

        :param str value: (optional) The entity value that was recognized in the
               user input.
        :param float confidence: (optional) A decimal percentage that represents
               Watson's confidence in the recognized entity.
        """
        self.value = value
        self.confidence = confidence

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'RuntimeEntityAlternative':
        """Initialize a RuntimeEntityAlternative object from a json dictionary."""
        args = {}
        if 'value' in _dict:
            args['value'] = _dict.get('value')
        if 'confidence' in _dict:
            args['confidence'] = _dict.get('confidence')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeEntityAlternative object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'value') and self.value is not None:
            _dict['value'] = self.value
        if hasattr(self, 'confidence') and self.confidence is not None:
            _dict['confidence'] = self.confidence
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeEntityAlternative object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'RuntimeEntityAlternative') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'RuntimeEntityAlternative') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RuntimeEntityInterpretation():
    """
    RuntimeEntityInterpretation.

    :attr str calendar_type: (optional) The calendar used to represent a recognized
          date (for example, `Gregorian`).
    :attr str datetime_link: (optional) A unique identifier used to associate a
          recognized time and date. If the user input contains a date and time that are
          mentioned together (for example, `Today at 5`, the same **datetime_link** value
          is returned for both the `@sys-date` and `@sys-time` entities).
    :attr str festival: (optional) A locale-specific holiday name (such as
          `thanksgiving` or `christmas`). This property is included when a `@sys-date`
          entity is recognized based on a holiday name in the user input.
    :attr str granularity: (optional) The precision or duration of a time range
          specified by a recognized `@sys-time` or `@sys-date` entity.
    :attr str range_link: (optional) A unique identifier used to associate multiple
          recognized `@sys-date`, `@sys-time`, or `@sys-number` entities that are
          recognized as a range of values in the user's input (for example, `from July 4
          until July 14` or `from 20 to 25`).
    :attr str range_modifier: (optional) The word in the user input that indicates
          that a `sys-date` or `sys-time` entity is part of an implied range where only
          one date or time is specified (for example, `since` or `until`).
    :attr float relative_day: (optional) A recognized mention of a relative day,
          represented numerically as an offset from the current date (for example, `-1`
          for `yesterday` or `10` for `in ten days`).
    :attr float relative_month: (optional) A recognized mention of a relative month,
          represented numerically as an offset from the current month (for example, `1`
          for `next month` or `-3` for `three months ago`).
    :attr float relative_week: (optional) A recognized mention of a relative week,
          represented numerically as an offset from the current week (for example, `2` for
          `in two weeks` or `-1` for `last week).
    :attr float relative_weekend: (optional) A recognized mention of a relative date
          range for a weekend, represented numerically as an offset from the current
          weekend (for example, `0` for `this weekend` or `-1` for `last weekend`).
    :attr float relative_year: (optional) A recognized mention of a relative year,
          represented numerically as an offset from the current year (for example, `1` for
          `next year` or `-5` for `five years ago`).
    :attr float specific_day: (optional) A recognized mention of a specific date,
          represented numerically as the date within the month (for example, `30` for
          `June 30`.).
    :attr str specific_day_of_week: (optional) A recognized mention of a specific
          day of the week as a lowercase string (for example, `monday`).
    :attr float specific_month: (optional) A recognized mention of a specific month,
          represented numerically (for example, `7` for `July`).
    :attr float specific_quarter: (optional) A recognized mention of a specific
          quarter, represented numerically (for example, `3` for `the third quarter`).
    :attr float specific_year: (optional) A recognized mention of a specific year
          (for example, `2016`).
    :attr float numeric_value: (optional) A recognized numeric value, represented as
          an integer or double.
    :attr str subtype: (optional) The type of numeric value recognized in the user
          input (`integer` or `rational`).
    :attr str part_of_day: (optional) A recognized term for a time that was
          mentioned as a part of the day in the user's input (for example, `morning` or
          `afternoon`).
    :attr float relative_hour: (optional) A recognized mention of a relative hour,
          represented numerically as an offset from the current hour (for example, `3` for
          `in three hours` or `-1` for `an hour ago`).
    :attr float relative_minute: (optional) A recognized mention of a relative time,
          represented numerically as an offset in minutes from the current time (for
          example, `5` for `in five minutes` or `-15` for `fifteen minutes ago`).
    :attr float relative_second: (optional) A recognized mention of a relative time,
          represented numerically as an offset in seconds from the current time (for
          example, `10` for `in ten seconds` or `-30` for `thirty seconds ago`).
    :attr float specific_hour: (optional) A recognized specific hour mentioned as
          part of a time value (for example, `10` for `10:15 AM`.).
    :attr float specific_minute: (optional) A recognized specific minute mentioned
          as part of a time value (for example, `15` for `10:15 AM`.).
    :attr float specific_second: (optional) A recognized specific second mentioned
          as part of a time value (for example, `30` for `10:15:30 AM`.).
    :attr str timezone: (optional) A recognized time zone mentioned as part of a
          time value (for example, `EST`).
    """

    def __init__(self,
                 *,
                 calendar_type: str = None,
                 datetime_link: str = None,
                 festival: str = None,
                 granularity: str = None,
                 range_link: str = None,
                 range_modifier: str = None,
                 relative_day: float = None,
                 relative_month: float = None,
                 relative_week: float = None,
                 relative_weekend: float = None,
                 relative_year: float = None,
                 specific_day: float = None,
                 specific_day_of_week: str = None,
                 specific_month: float = None,
                 specific_quarter: float = None,
                 specific_year: float = None,
                 numeric_value: float = None,
                 subtype: str = None,
                 part_of_day: str = None,
                 relative_hour: float = None,
                 relative_minute: float = None,
                 relative_second: float = None,
                 specific_hour: float = None,
                 specific_minute: float = None,
                 specific_second: float = None,
                 timezone: str = None) -> None:
        """
        Initialize a RuntimeEntityInterpretation object.

        :param str calendar_type: (optional) The calendar used to represent a
               recognized date (for example, `Gregorian`).
        :param str datetime_link: (optional) A unique identifier used to associate
               a recognized time and date. If the user input contains a date and time that
               are mentioned together (for example, `Today at 5`, the same
               **datetime_link** value is returned for both the `@sys-date` and
               `@sys-time` entities).
        :param str festival: (optional) A locale-specific holiday name (such as
               `thanksgiving` or `christmas`). This property is included when a
               `@sys-date` entity is recognized based on a holiday name in the user input.
        :param str granularity: (optional) The precision or duration of a time
               range specified by a recognized `@sys-time` or `@sys-date` entity.
        :param str range_link: (optional) A unique identifier used to associate
               multiple recognized `@sys-date`, `@sys-time`, or `@sys-number` entities
               that are recognized as a range of values in the user's input (for example,
               `from July 4 until July 14` or `from 20 to 25`).
        :param str range_modifier: (optional) The word in the user input that
               indicates that a `sys-date` or `sys-time` entity is part of an implied
               range where only one date or time is specified (for example, `since` or
               `until`).
        :param float relative_day: (optional) A recognized mention of a relative
               day, represented numerically as an offset from the current date (for
               example, `-1` for `yesterday` or `10` for `in ten days`).
        :param float relative_month: (optional) A recognized mention of a relative
               month, represented numerically as an offset from the current month (for
               example, `1` for `next month` or `-3` for `three months ago`).
        :param float relative_week: (optional) A recognized mention of a relative
               week, represented numerically as an offset from the current week (for
               example, `2` for `in two weeks` or `-1` for `last week).
        :param float relative_weekend: (optional) A recognized mention of a
               relative date range for a weekend, represented numerically as an offset
               from the current weekend (for example, `0` for `this weekend` or `-1` for
               `last weekend`).
        :param float relative_year: (optional) A recognized mention of a relative
               year, represented numerically as an offset from the current year (for
               example, `1` for `next year` or `-5` for `five years ago`).
        :param float specific_day: (optional) A recognized mention of a specific
               date, represented numerically as the date within the month (for example,
               `30` for `June 30`.).
        :param str specific_day_of_week: (optional) A recognized mention of a
               specific day of the week as a lowercase string (for example, `monday`).
        :param float specific_month: (optional) A recognized mention of a specific
               month, represented numerically (for example, `7` for `July`).
        :param float specific_quarter: (optional) A recognized mention of a
               specific quarter, represented numerically (for example, `3` for `the third
               quarter`).
        :param float specific_year: (optional) A recognized mention of a specific
               year (for example, `2016`).
        :param float numeric_value: (optional) A recognized numeric value,
               represented as an integer or double.
        :param str subtype: (optional) The type of numeric value recognized in the
               user input (`integer` or `rational`).
        :param str part_of_day: (optional) A recognized term for a time that was
               mentioned as a part of the day in the user's input (for example, `morning`
               or `afternoon`).
        :param float relative_hour: (optional) A recognized mention of a relative
               hour, represented numerically as an offset from the current hour (for
               example, `3` for `in three hours` or `-1` for `an hour ago`).
        :param float relative_minute: (optional) A recognized mention of a relative
               time, represented numerically as an offset in minutes from the current time
               (for example, `5` for `in five minutes` or `-15` for `fifteen minutes
               ago`).
        :param float relative_second: (optional) A recognized mention of a relative
               time, represented numerically as an offset in seconds from the current time
               (for example, `10` for `in ten seconds` or `-30` for `thirty seconds ago`).
        :param float specific_hour: (optional) A recognized specific hour mentioned
               as part of a time value (for example, `10` for `10:15 AM`.).
        :param float specific_minute: (optional) A recognized specific minute
               mentioned as part of a time value (for example, `15` for `10:15 AM`.).
        :param float specific_second: (optional) A recognized specific second
               mentioned as part of a time value (for example, `30` for `10:15:30 AM`.).
        :param str timezone: (optional) A recognized time zone mentioned as part of
               a time value (for example, `EST`).
        """
        self.calendar_type = calendar_type
        self.datetime_link = datetime_link
        self.festival = festival
        self.granularity = granularity
        self.range_link = range_link
        self.range_modifier = range_modifier
        self.relative_day = relative_day
        self.relative_month = relative_month
        self.relative_week = relative_week
        self.relative_weekend = relative_weekend
        self.relative_year = relative_year
        self.specific_day = specific_day
        self.specific_day_of_week = specific_day_of_week
        self.specific_month = specific_month
        self.specific_quarter = specific_quarter
        self.specific_year = specific_year
        self.numeric_value = numeric_value
        self.subtype = subtype
        self.part_of_day = part_of_day
        self.relative_hour = relative_hour
        self.relative_minute = relative_minute
        self.relative_second = relative_second
        self.specific_hour = specific_hour
        self.specific_minute = specific_minute
        self.specific_second = specific_second
        self.timezone = timezone

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'RuntimeEntityInterpretation':
        """Initialize a RuntimeEntityInterpretation object from a json dictionary."""
        args = {}
        if 'calendar_type' in _dict:
            args['calendar_type'] = _dict.get('calendar_type')
        if 'datetime_link' in _dict:
            args['datetime_link'] = _dict.get('datetime_link')
        if 'festival' in _dict:
            args['festival'] = _dict.get('festival')
        if 'granularity' in _dict:
            args['granularity'] = _dict.get('granularity')
        if 'range_link' in _dict:
            args['range_link'] = _dict.get('range_link')
        if 'range_modifier' in _dict:
            args['range_modifier'] = _dict.get('range_modifier')
        if 'relative_day' in _dict:
            args['relative_day'] = _dict.get('relative_day')
        if 'relative_month' in _dict:
            args['relative_month'] = _dict.get('relative_month')
        if 'relative_week' in _dict:
            args['relative_week'] = _dict.get('relative_week')
        if 'relative_weekend' in _dict:
            args['relative_weekend'] = _dict.get('relative_weekend')
        if 'relative_year' in _dict:
            args['relative_year'] = _dict.get('relative_year')
        if 'specific_day' in _dict:
            args['specific_day'] = _dict.get('specific_day')
        if 'specific_day_of_week' in _dict:
            args['specific_day_of_week'] = _dict.get('specific_day_of_week')
        if 'specific_month' in _dict:
            args['specific_month'] = _dict.get('specific_month')
        if 'specific_quarter' in _dict:
            args['specific_quarter'] = _dict.get('specific_quarter')
        if 'specific_year' in _dict:
            args['specific_year'] = _dict.get('specific_year')
        if 'numeric_value' in _dict:
            args['numeric_value'] = _dict.get('numeric_value')
        if 'subtype' in _dict:
            args['subtype'] = _dict.get('subtype')
        if 'part_of_day' in _dict:
            args['part_of_day'] = _dict.get('part_of_day')
        if 'relative_hour' in _dict:
            args['relative_hour'] = _dict.get('relative_hour')
        if 'relative_minute' in _dict:
            args['relative_minute'] = _dict.get('relative_minute')
        if 'relative_second' in _dict:
            args['relative_second'] = _dict.get('relative_second')
        if 'specific_hour' in _dict:
            args['specific_hour'] = _dict.get('specific_hour')
        if 'specific_minute' in _dict:
            args['specific_minute'] = _dict.get('specific_minute')
        if 'specific_second' in _dict:
            args['specific_second'] = _dict.get('specific_second')
        if 'timezone' in _dict:
            args['timezone'] = _dict.get('timezone')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeEntityInterpretation object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'calendar_type') and self.calendar_type is not None:
            _dict['calendar_type'] = self.calendar_type
        if hasattr(self, 'datetime_link') and self.datetime_link is not None:
            _dict['datetime_link'] = self.datetime_link
        if hasattr(self, 'festival') and self.festival is not None:
            _dict['festival'] = self.festival
        if hasattr(self, 'granularity') and self.granularity is not None:
            _dict['granularity'] = self.granularity
        if hasattr(self, 'range_link') and self.range_link is not None:
            _dict['range_link'] = self.range_link
        if hasattr(self, 'range_modifier') and self.range_modifier is not None:
            _dict['range_modifier'] = self.range_modifier
        if hasattr(self, 'relative_day') and self.relative_day is not None:
            _dict['relative_day'] = self.relative_day
        if hasattr(self, 'relative_month') and self.relative_month is not None:
            _dict['relative_month'] = self.relative_month
        if hasattr(self, 'relative_week') and self.relative_week is not None:
            _dict['relative_week'] = self.relative_week
        if hasattr(self,
                   'relative_weekend') and self.relative_weekend is not None:
            _dict['relative_weekend'] = self.relative_weekend
        if hasattr(self, 'relative_year') and self.relative_year is not None:
            _dict['relative_year'] = self.relative_year
        if hasattr(self, 'specific_day') and self.specific_day is not None:
            _dict['specific_day'] = self.specific_day
        if hasattr(self, 'specific_day_of_week'
                  ) and self.specific_day_of_week is not None:
            _dict['specific_day_of_week'] = self.specific_day_of_week
        if hasattr(self, 'specific_month') and self.specific_month is not None:
            _dict['specific_month'] = self.specific_month
        if hasattr(self,
                   'specific_quarter') and self.specific_quarter is not None:
            _dict['specific_quarter'] = self.specific_quarter
        if hasattr(self, 'specific_year') and self.specific_year is not None:
            _dict['specific_year'] = self.specific_year
        if hasattr(self, 'numeric_value') and self.numeric_value is not None:
            _dict['numeric_value'] = self.numeric_value
        if hasattr(self, 'subtype') and self.subtype is not None:
            _dict['subtype'] = self.subtype
        if hasattr(self, 'part_of_day') and self.part_of_day is not None:
            _dict['part_of_day'] = self.part_of_day
        if hasattr(self, 'relative_hour') and self.relative_hour is not None:
            _dict['relative_hour'] = self.relative_hour
        if hasattr(self,
                   'relative_minute') and self.relative_minute is not None:
            _dict['relative_minute'] = self.relative_minute
        if hasattr(self,
                   'relative_second') and self.relative_second is not None:
            _dict['relative_second'] = self.relative_second
        if hasattr(self, 'specific_hour') and self.specific_hour is not None:
            _dict['specific_hour'] = self.specific_hour
        if hasattr(self,
                   'specific_minute') and self.specific_minute is not None:
            _dict['specific_minute'] = self.specific_minute
        if hasattr(self,
                   'specific_second') and self.specific_second is not None:
            _dict['specific_second'] = self.specific_second
        if hasattr(self, 'timezone') and self.timezone is not None:
            _dict['timezone'] = self.timezone
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeEntityInterpretation object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'RuntimeEntityInterpretation') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'RuntimeEntityInterpretation') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class GranularityEnum(str, Enum):
        """
        The precision or duration of a time range specified by a recognized `@sys-time` or
        `@sys-date` entity.
        """
        DAY = 'day'
        FORTNIGHT = 'fortnight'
        HOUR = 'hour'
        INSTANT = 'instant'
        MINUTE = 'minute'
        MONTH = 'month'
        QUARTER = 'quarter'
        SECOND = 'second'
        WEEK = 'week'
        WEEKEND = 'weekend'
        YEAR = 'year'


class RuntimeEntityRole():
    """
    An object describing the role played by a system entity that is specifies the
    beginning or end of a range recognized in the user input. This property is included
    only if the new system entities are enabled for the skill.

    :attr str type: (optional) The relationship of the entity to the range.
    """

    def __init__(self, *, type: str = None) -> None:
        """
        Initialize a RuntimeEntityRole object.

        :param str type: (optional) The relationship of the entity to the range.
        """
        self.type = type

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'RuntimeEntityRole':
        """Initialize a RuntimeEntityRole object from a json dictionary."""
        args = {}
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeEntityRole object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeEntityRole object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'RuntimeEntityRole') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'RuntimeEntityRole') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class TypeEnum(str, Enum):
        """
        The relationship of the entity to the range.
        """
        DATE_FROM = 'date_from'
        DATE_TO = 'date_to'
        NUMBER_FROM = 'number_from'
        NUMBER_TO = 'number_to'
        TIME_FROM = 'time_from'
        TIME_TO = 'time_to'


class RuntimeIntent():
    """
    An intent identified in the user input.

    :attr str intent: The name of the recognized intent.
    :attr float confidence: A decimal percentage that represents Watson's confidence
          in the intent.
    """

    def __init__(self, intent: str, confidence: float) -> None:
        """
        Initialize a RuntimeIntent object.

        :param str intent: The name of the recognized intent.
        :param float confidence: A decimal percentage that represents Watson's
               confidence in the intent.
        """
        self.intent = intent
        self.confidence = confidence

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'RuntimeIntent':
        """Initialize a RuntimeIntent object from a json dictionary."""
        args = {}
        if 'intent' in _dict:
            args['intent'] = _dict.get('intent')
        else:
            raise ValueError(
                'Required property \'intent\' not present in RuntimeIntent JSON'
            )
        if 'confidence' in _dict:
            args['confidence'] = _dict.get('confidence')
        else:
            raise ValueError(
                'Required property \'confidence\' not present in RuntimeIntent JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeIntent object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'intent') and self.intent is not None:
            _dict['intent'] = self.intent
        if hasattr(self, 'confidence') and self.confidence is not None:
            _dict['confidence'] = self.confidence
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeIntent object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'RuntimeIntent') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'RuntimeIntent') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RuntimeResponseGeneric():
    """
    RuntimeResponseGeneric.

    """

    def __init__(self) -> None:
        """
        Initialize a RuntimeResponseGeneric object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
            ", ".join([
                'RuntimeResponseGenericRuntimeResponseTypeText',
                'RuntimeResponseGenericRuntimeResponseTypePause',
                'RuntimeResponseGenericRuntimeResponseTypeImage',
                'RuntimeResponseGenericRuntimeResponseTypeOption',
                'RuntimeResponseGenericRuntimeResponseTypeConnectToAgent',
                'RuntimeResponseGenericRuntimeResponseTypeSuggestion',
                'RuntimeResponseGenericRuntimeResponseTypeChannelTransfer',
                'RuntimeResponseGenericRuntimeResponseTypeSearch',
                'RuntimeResponseGenericRuntimeResponseTypeUserDefined',
                'RuntimeResponseGenericRuntimeResponseTypeVideo',
                'RuntimeResponseGenericRuntimeResponseTypeAudio',
                'RuntimeResponseGenericRuntimeResponseTypeIframe'
            ]))
        raise Exception(msg)

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'RuntimeResponseGeneric':
        """Initialize a RuntimeResponseGeneric object from a json dictionary."""
        disc_class = cls._get_class_by_discriminator(_dict)
        if disc_class != cls:
            return disc_class.from_dict(_dict)
        msg = (
            "Cannot convert dictionary into an instance of base class 'RuntimeResponseGeneric'. "
            + "The discriminator value should map to a valid subclass: {1}"
        ).format(", ".join([
            'RuntimeResponseGenericRuntimeResponseTypeText',
            'RuntimeResponseGenericRuntimeResponseTypePause',
            'RuntimeResponseGenericRuntimeResponseTypeImage',
            'RuntimeResponseGenericRuntimeResponseTypeOption',
            'RuntimeResponseGenericRuntimeResponseTypeConnectToAgent',
            'RuntimeResponseGenericRuntimeResponseTypeSuggestion',
            'RuntimeResponseGenericRuntimeResponseTypeChannelTransfer',
            'RuntimeResponseGenericRuntimeResponseTypeSearch',
            'RuntimeResponseGenericRuntimeResponseTypeUserDefined',
            'RuntimeResponseGenericRuntimeResponseTypeVideo',
            'RuntimeResponseGenericRuntimeResponseTypeAudio',
            'RuntimeResponseGenericRuntimeResponseTypeIframe'
        ]))
        raise Exception(msg)

    @classmethod
    def _from_dict(cls, _dict: Dict):
        """Initialize a RuntimeResponseGeneric object from a json dictionary."""
        return cls.from_dict(_dict)

    @classmethod
    def _get_class_by_discriminator(cls, _dict: Dict) -> object:
        mapping = {}
        mapping['audio'] = 'RuntimeResponseGenericRuntimeResponseTypeAudio'
        mapping[
            'channel_transfer'] = 'RuntimeResponseGenericRuntimeResponseTypeChannelTransfer'
        mapping[
            'connect_to_agent'] = 'RuntimeResponseGenericRuntimeResponseTypeConnectToAgent'
        mapping['iframe'] = 'RuntimeResponseGenericRuntimeResponseTypeIframe'
        mapping['image'] = 'RuntimeResponseGenericRuntimeResponseTypeImage'
        mapping['option'] = 'RuntimeResponseGenericRuntimeResponseTypeOption'
        mapping[
            'suggestion'] = 'RuntimeResponseGenericRuntimeResponseTypeSuggestion'
        mapping['pause'] = 'RuntimeResponseGenericRuntimeResponseTypePause'
        mapping['search'] = 'RuntimeResponseGenericRuntimeResponseTypeSearch'
        mapping['text'] = 'RuntimeResponseGenericRuntimeResponseTypeText'
        mapping[
            'user_defined'] = 'RuntimeResponseGenericRuntimeResponseTypeUserDefined'
        mapping['video'] = 'RuntimeResponseGenericRuntimeResponseTypeVideo'
        disc_value = _dict.get('response_type')
        if disc_value is None:
            raise ValueError(
                'Discriminator property \'response_type\' not found in RuntimeResponseGeneric JSON'
            )
        class_name = mapping.get(disc_value, disc_value)
        try:
            disc_class = getattr(sys.modules[__name__], class_name)
        except AttributeError:
            disc_class = cls
        if isinstance(disc_class, object):
            return disc_class
        raise TypeError('%s is not a discriminator class' % class_name)


class SearchResult():
    """
    SearchResult.

    :attr str id: The unique identifier of the document in the Discovery service
          collection.
          This property is included in responses from search skills, which are available
          only to Plus or Enterprise plan users.
    :attr SearchResultMetadata result_metadata: An object containing search result
          metadata from the Discovery service.
    :attr str body: (optional) A description of the search result. This is taken
          from an abstract, summary, or highlight field in the Discovery service response,
          as specified in the search skill configuration.
    :attr str title: (optional) The title of the search result. This is taken from a
          title or name field in the Discovery service response, as specified in the
          search skill configuration.
    :attr str url: (optional) The URL of the original data object in its native data
          source.
    :attr SearchResultHighlight highlight: (optional) An object containing segments
          of text from search results with query-matching text highlighted using HTML
          `<em>` tags.
    :attr List[SearchResultAnswer] answers: (optional) An array specifying segments
          of text within the result that were identified as direct answers to the search
          query. Currently, only the single answer with the highest confidence (if any) is
          returned.
          **Note:** This property uses the answer finding beta feature, and is available
          only if the search skill is connected to a Discovery v2 service instance.
    """

    def __init__(self,
                 id: str,
                 result_metadata: 'SearchResultMetadata',
                 *,
                 body: str = None,
                 title: str = None,
                 url: str = None,
                 highlight: 'SearchResultHighlight' = None,
                 answers: List['SearchResultAnswer'] = None) -> None:
        """
        Initialize a SearchResult object.

        :param str id: The unique identifier of the document in the Discovery
               service collection.
               This property is included in responses from search skills, which are
               available only to Plus or Enterprise plan users.
        :param SearchResultMetadata result_metadata: An object containing search
               result metadata from the Discovery service.
        :param str body: (optional) A description of the search result. This is
               taken from an abstract, summary, or highlight field in the Discovery
               service response, as specified in the search skill configuration.
        :param str title: (optional) The title of the search result. This is taken
               from a title or name field in the Discovery service response, as specified
               in the search skill configuration.
        :param str url: (optional) The URL of the original data object in its
               native data source.
        :param SearchResultHighlight highlight: (optional) An object containing
               segments of text from search results with query-matching text highlighted
               using HTML `<em>` tags.
        :param List[SearchResultAnswer] answers: (optional) An array specifying
               segments of text within the result that were identified as direct answers
               to the search query. Currently, only the single answer with the highest
               confidence (if any) is returned.
               **Note:** This property uses the answer finding beta feature, and is
               available only if the search skill is connected to a Discovery v2 service
               instance.
        """
        self.id = id
        self.result_metadata = result_metadata
        self.body = body
        self.title = title
        self.url = url
        self.highlight = highlight
        self.answers = answers

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SearchResult':
        """Initialize a SearchResult object from a json dictionary."""
        args = {}
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError(
                'Required property \'id\' not present in SearchResult JSON')
        if 'result_metadata' in _dict:
            args['result_metadata'] = SearchResultMetadata.from_dict(
                _dict.get('result_metadata'))
        else:
            raise ValueError(
                'Required property \'result_metadata\' not present in SearchResult JSON'
            )
        if 'body' in _dict:
            args['body'] = _dict.get('body')
        if 'title' in _dict:
            args['title'] = _dict.get('title')
        if 'url' in _dict:
            args['url'] = _dict.get('url')
        if 'highlight' in _dict:
            args['highlight'] = SearchResultHighlight.from_dict(
                _dict.get('highlight'))
        if 'answers' in _dict:
            args['answers'] = [
                SearchResultAnswer.from_dict(x) for x in _dict.get('answers')
            ]
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a SearchResult object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self,
                   'result_metadata') and self.result_metadata is not None:
            _dict['result_metadata'] = self.result_metadata.to_dict()
        if hasattr(self, 'body') and self.body is not None:
            _dict['body'] = self.body
        if hasattr(self, 'title') and self.title is not None:
            _dict['title'] = self.title
        if hasattr(self, 'url') and self.url is not None:
            _dict['url'] = self.url
        if hasattr(self, 'highlight') and self.highlight is not None:
            _dict['highlight'] = self.highlight.to_dict()
        if hasattr(self, 'answers') and self.answers is not None:
            _dict['answers'] = [x.to_dict() for x in self.answers]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this SearchResult object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'SearchResult') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'SearchResult') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class SearchResultAnswer():
    """
    An object specifing a segment of text that was identified as a direct answer to the
    search query.

    :attr str text: The text of the answer.
    :attr float confidence: The confidence score for the answer, as returned by the
          Discovery service.
    """

    def __init__(self, text: str, confidence: float) -> None:
        """
        Initialize a SearchResultAnswer object.

        :param str text: The text of the answer.
        :param float confidence: The confidence score for the answer, as returned
               by the Discovery service.
        """
        self.text = text
        self.confidence = confidence

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SearchResultAnswer':
        """Initialize a SearchResultAnswer object from a json dictionary."""
        args = {}
        if 'text' in _dict:
            args['text'] = _dict.get('text')
        else:
            raise ValueError(
                'Required property \'text\' not present in SearchResultAnswer JSON'
            )
        if 'confidence' in _dict:
            args['confidence'] = _dict.get('confidence')
        else:
            raise ValueError(
                'Required property \'confidence\' not present in SearchResultAnswer JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a SearchResultAnswer object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'text') and self.text is not None:
            _dict['text'] = self.text
        if hasattr(self, 'confidence') and self.confidence is not None:
            _dict['confidence'] = self.confidence
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this SearchResultAnswer object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'SearchResultAnswer') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'SearchResultAnswer') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class SearchResultHighlight():
    """
    An object containing segments of text from search results with query-matching text
    highlighted using HTML `<em>` tags.

    :attr List[str] body: (optional) An array of strings containing segments taken
          from body text in the search results, with query-matching substrings
          highlighted.
    :attr List[str] title: (optional) An array of strings containing segments taken
          from title text in the search results, with query-matching substrings
          highlighted.
    :attr List[str] url: (optional) An array of strings containing segments taken
          from URLs in the search results, with query-matching substrings highlighted.
    """

    # The set of defined properties for the class
    _properties = frozenset(['body', 'title', 'url'])

    def __init__(self,
                 *,
                 body: List[str] = None,
                 title: List[str] = None,
                 url: List[str] = None,
                 **kwargs) -> None:
        """
        Initialize a SearchResultHighlight object.

        :param List[str] body: (optional) An array of strings containing segments
               taken from body text in the search results, with query-matching substrings
               highlighted.
        :param List[str] title: (optional) An array of strings containing segments
               taken from title text in the search results, with query-matching substrings
               highlighted.
        :param List[str] url: (optional) An array of strings containing segments
               taken from URLs in the search results, with query-matching substrings
               highlighted.
        :param **kwargs: (optional) Any additional properties.
        """
        self.body = body
        self.title = title
        self.url = url
        for _key, _value in kwargs.items():
            setattr(self, _key, _value)

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SearchResultHighlight':
        """Initialize a SearchResultHighlight object from a json dictionary."""
        args = {}
        if 'body' in _dict:
            args['body'] = _dict.get('body')
        if 'title' in _dict:
            args['title'] = _dict.get('title')
        if 'url' in _dict:
            args['url'] = _dict.get('url')
        args.update(
            {k: v for (k, v) in _dict.items() if k not in cls._properties})
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a SearchResultHighlight object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'body') and self.body is not None:
            _dict['body'] = self.body
        if hasattr(self, 'title') and self.title is not None:
            _dict['title'] = self.title
        if hasattr(self, 'url') and self.url is not None:
            _dict['url'] = self.url
        for _key in [
                k for k in vars(self).keys()
                if k not in SearchResultHighlight._properties
        ]:
            if getattr(self, _key, None) is not None:
                _dict[_key] = getattr(self, _key)
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def get_properties(self) -> Dict:
        """Return a dictionary of arbitrary properties from this instance of SearchResultHighlight"""
        _dict = {}

        for _key in [
                k for k in vars(self).keys()
                if k not in SearchResultHighlight._properties
        ]:
            _dict[_key] = getattr(self, _key)
        return _dict

    def set_properties(self, _dict: dict):
        """Set a dictionary of arbitrary properties to this instance of SearchResultHighlight"""
        for _key in [
                k for k in vars(self).keys()
                if k not in SearchResultHighlight._properties
        ]:
            delattr(self, _key)

        for _key, _value in _dict.items():
            if _key not in SearchResultHighlight._properties:
                setattr(self, _key, _value)

    def __str__(self) -> str:
        """Return a `str` version of this SearchResultHighlight object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'SearchResultHighlight') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'SearchResultHighlight') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class SearchResultMetadata():
    """
    An object containing search result metadata from the Discovery service.

    :attr float confidence: (optional) The confidence score for the given result, as
          returned by the Discovery service.
    :attr float score: (optional) An unbounded measure of the relevance of a
          particular result, dependent on the query and matching document. A higher score
          indicates a greater match to the query parameters.
    """

    def __init__(self,
                 *,
                 confidence: float = None,
                 score: float = None) -> None:
        """
        Initialize a SearchResultMetadata object.

        :param float confidence: (optional) The confidence score for the given
               result, as returned by the Discovery service.
        :param float score: (optional) An unbounded measure of the relevance of a
               particular result, dependent on the query and matching document. A higher
               score indicates a greater match to the query parameters.
        """
        self.confidence = confidence
        self.score = score

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SearchResultMetadata':
        """Initialize a SearchResultMetadata object from a json dictionary."""
        args = {}
        if 'confidence' in _dict:
            args['confidence'] = _dict.get('confidence')
        if 'score' in _dict:
            args['score'] = _dict.get('score')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a SearchResultMetadata object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'confidence') and self.confidence is not None:
            _dict['confidence'] = self.confidence
        if hasattr(self, 'score') and self.score is not None:
            _dict['score'] = self.score
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this SearchResultMetadata object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'SearchResultMetadata') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'SearchResultMetadata') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class SessionResponse():
    """
    SessionResponse.

    :attr str session_id: The session ID.
    """

    def __init__(self, session_id: str) -> None:
        """
        Initialize a SessionResponse object.

        :param str session_id: The session ID.
        """
        self.session_id = session_id

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'SessionResponse':
        """Initialize a SessionResponse object from a json dictionary."""
        args = {}
        if 'session_id' in _dict:
            args['session_id'] = _dict.get('session_id')
        else:
            raise ValueError(
                'Required property \'session_id\' not present in SessionResponse JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a SessionResponse object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'session_id') and self.session_id is not None:
            _dict['session_id'] = self.session_id
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this SessionResponse object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'SessionResponse') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'SessionResponse') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LogMessageSourceAction(LogMessageSource):
    """
    An object that identifies the dialog element that generated the error message.

    :attr str type: A string that indicates the type of dialog element that
          generated the error message.
    :attr str action: The unique identifier of the action that generated the error
          message.
    """

    def __init__(self, type: str, action: str) -> None:
        """
        Initialize a LogMessageSourceAction object.

        :param str type: A string that indicates the type of dialog element that
               generated the error message.
        :param str action: The unique identifier of the action that generated the
               error message.
        """
        # pylint: disable=super-init-not-called
        self.type = type
        self.action = action

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LogMessageSourceAction':
        """Initialize a LogMessageSourceAction object from a json dictionary."""
        args = {}
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in LogMessageSourceAction JSON'
            )
        if 'action' in _dict:
            args['action'] = _dict.get('action')
        else:
            raise ValueError(
                'Required property \'action\' not present in LogMessageSourceAction JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LogMessageSourceAction object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        if hasattr(self, 'action') and self.action is not None:
            _dict['action'] = self.action
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LogMessageSourceAction object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LogMessageSourceAction') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LogMessageSourceAction') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LogMessageSourceDialogNode(LogMessageSource):
    """
    An object that identifies the dialog element that generated the error message.

    :attr str type: A string that indicates the type of dialog element that
          generated the error message.
    :attr str dialog_node: The unique identifier of the dialog node that generated
          the error message.
    """

    def __init__(self, type: str, dialog_node: str) -> None:
        """
        Initialize a LogMessageSourceDialogNode object.

        :param str type: A string that indicates the type of dialog element that
               generated the error message.
        :param str dialog_node: The unique identifier of the dialog node that
               generated the error message.
        """
        # pylint: disable=super-init-not-called
        self.type = type
        self.dialog_node = dialog_node

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LogMessageSourceDialogNode':
        """Initialize a LogMessageSourceDialogNode object from a json dictionary."""
        args = {}
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in LogMessageSourceDialogNode JSON'
            )
        if 'dialog_node' in _dict:
            args['dialog_node'] = _dict.get('dialog_node')
        else:
            raise ValueError(
                'Required property \'dialog_node\' not present in LogMessageSourceDialogNode JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LogMessageSourceDialogNode object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        if hasattr(self, 'dialog_node') and self.dialog_node is not None:
            _dict['dialog_node'] = self.dialog_node
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LogMessageSourceDialogNode object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LogMessageSourceDialogNode') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LogMessageSourceDialogNode') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LogMessageSourceHandler(LogMessageSource):
    """
    An object that identifies the dialog element that generated the error message.

    :attr str type: A string that indicates the type of dialog element that
          generated the error message.
    :attr str action: The unique identifier of the action that generated the error
          message.
    :attr str step: (optional) The unique identifier of the step that generated the
          error message.
    :attr str handler: The unique identifier of the handler that generated the error
          message.
    """

    def __init__(self,
                 type: str,
                 action: str,
                 handler: str,
                 *,
                 step: str = None) -> None:
        """
        Initialize a LogMessageSourceHandler object.

        :param str type: A string that indicates the type of dialog element that
               generated the error message.
        :param str action: The unique identifier of the action that generated the
               error message.
        :param str handler: The unique identifier of the handler that generated the
               error message.
        :param str step: (optional) The unique identifier of the step that
               generated the error message.
        """
        # pylint: disable=super-init-not-called
        self.type = type
        self.action = action
        self.step = step
        self.handler = handler

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LogMessageSourceHandler':
        """Initialize a LogMessageSourceHandler object from a json dictionary."""
        args = {}
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in LogMessageSourceHandler JSON'
            )
        if 'action' in _dict:
            args['action'] = _dict.get('action')
        else:
            raise ValueError(
                'Required property \'action\' not present in LogMessageSourceHandler JSON'
            )
        if 'step' in _dict:
            args['step'] = _dict.get('step')
        if 'handler' in _dict:
            args['handler'] = _dict.get('handler')
        else:
            raise ValueError(
                'Required property \'handler\' not present in LogMessageSourceHandler JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LogMessageSourceHandler object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        if hasattr(self, 'action') and self.action is not None:
            _dict['action'] = self.action
        if hasattr(self, 'step') and self.step is not None:
            _dict['step'] = self.step
        if hasattr(self, 'handler') and self.handler is not None:
            _dict['handler'] = self.handler
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LogMessageSourceHandler object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LogMessageSourceHandler') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LogMessageSourceHandler') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class LogMessageSourceStep(LogMessageSource):
    """
    An object that identifies the dialog element that generated the error message.

    :attr str type: A string that indicates the type of dialog element that
          generated the error message.
    :attr str action: The unique identifier of the action that generated the error
          message.
    :attr str step: The unique identifier of the step that generated the error
          message.
    """

    def __init__(self, type: str, action: str, step: str) -> None:
        """
        Initialize a LogMessageSourceStep object.

        :param str type: A string that indicates the type of dialog element that
               generated the error message.
        :param str action: The unique identifier of the action that generated the
               error message.
        :param str step: The unique identifier of the step that generated the error
               message.
        """
        # pylint: disable=super-init-not-called
        self.type = type
        self.action = action
        self.step = step

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'LogMessageSourceStep':
        """Initialize a LogMessageSourceStep object from a json dictionary."""
        args = {}
        if 'type' in _dict:
            args['type'] = _dict.get('type')
        else:
            raise ValueError(
                'Required property \'type\' not present in LogMessageSourceStep JSON'
            )
        if 'action' in _dict:
            args['action'] = _dict.get('action')
        else:
            raise ValueError(
                'Required property \'action\' not present in LogMessageSourceStep JSON'
            )
        if 'step' in _dict:
            args['step'] = _dict.get('step')
        else:
            raise ValueError(
                'Required property \'step\' not present in LogMessageSourceStep JSON'
            )
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a LogMessageSourceStep object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'type') and self.type is not None:
            _dict['type'] = self.type
        if hasattr(self, 'action') and self.action is not None:
            _dict['action'] = self.action
        if hasattr(self, 'step') and self.step is not None:
            _dict['step'] = self.step
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this LogMessageSourceStep object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'LogMessageSourceStep') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'LogMessageSourceStep') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RuntimeResponseGenericRuntimeResponseTypeAudio(RuntimeResponseGeneric):
    """
    RuntimeResponseGenericRuntimeResponseTypeAudio.

    :attr str response_type: The type of response returned by the dialog node. The
          specified response type must be supported by the client application or channel.
    :attr str source: The `https:` URL of the audio clip.
    :attr str title: (optional) The title or introductory text to show before the
          response.
    :attr str description: (optional) The description to show with the the response.
    :attr List[ResponseGenericChannel] channels: (optional) An array of objects
          specifying channels for which the response is intended. If **channels** is
          present, the response is intended for a built-in integration and should not be
          handled by an API client.
    :attr object channel_options: (optional) For internal use only.
    :attr str alt_text: (optional) Descriptive text that can be used for screen
          readers or other situations where the audio player cannot be seen.
    """

    def __init__(self,
                 response_type: str,
                 source: str,
                 *,
                 title: str = None,
                 description: str = None,
                 channels: List['ResponseGenericChannel'] = None,
                 channel_options: object = None,
                 alt_text: str = None) -> None:
        """
        Initialize a RuntimeResponseGenericRuntimeResponseTypeAudio object.

        :param str response_type: The type of response returned by the dialog node.
               The specified response type must be supported by the client application or
               channel.
        :param str source: The `https:` URL of the audio clip.
        :param str title: (optional) The title or introductory text to show before
               the response.
        :param str description: (optional) The description to show with the the
               response.
        :param List[ResponseGenericChannel] channels: (optional) An array of
               objects specifying channels for which the response is intended. If
               **channels** is present, the response is intended for a built-in
               integration and should not be handled by an API client.
        :param object channel_options: (optional) For internal use only.
        :param str alt_text: (optional) Descriptive text that can be used for
               screen readers or other situations where the audio player cannot be seen.
        """
        # pylint: disable=super-init-not-called
        self.response_type = response_type
        self.source = source
        self.title = title
        self.description = description
        self.channels = channels
        self.channel_options = channel_options
        self.alt_text = alt_text

    @classmethod
    def from_dict(
            cls,
            _dict: Dict) -> 'RuntimeResponseGenericRuntimeResponseTypeAudio':
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeAudio object from a json dictionary."""
        args = {}
        if 'response_type' in _dict:
            args['response_type'] = _dict.get('response_type')
        else:
            raise ValueError(
                'Required property \'response_type\' not present in RuntimeResponseGenericRuntimeResponseTypeAudio JSON'
            )
        if 'source' in _dict:
            args['source'] = _dict.get('source')
        else:
            raise ValueError(
                'Required property \'source\' not present in RuntimeResponseGenericRuntimeResponseTypeAudio JSON'
            )
        if 'title' in _dict:
            args['title'] = _dict.get('title')
        if 'description' in _dict:
            args['description'] = _dict.get('description')
        if 'channels' in _dict:
            args['channels'] = [
                ResponseGenericChannel.from_dict(x)
                for x in _dict.get('channels')
            ]
        if 'channel_options' in _dict:
            args['channel_options'] = _dict.get('channel_options')
        if 'alt_text' in _dict:
            args['alt_text'] = _dict.get('alt_text')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeAudio object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'response_type') and self.response_type is not None:
            _dict['response_type'] = self.response_type
        if hasattr(self, 'source') and self.source is not None:
            _dict['source'] = self.source
        if hasattr(self, 'title') and self.title is not None:
            _dict['title'] = self.title
        if hasattr(self, 'description') and self.description is not None:
            _dict['description'] = self.description
        if hasattr(self, 'channels') and self.channels is not None:
            _dict['channels'] = [x.to_dict() for x in self.channels]
        if hasattr(self,
                   'channel_options') and self.channel_options is not None:
            _dict['channel_options'] = self.channel_options
        if hasattr(self, 'alt_text') and self.alt_text is not None:
            _dict['alt_text'] = self.alt_text
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeResponseGenericRuntimeResponseTypeAudio object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self,
               other: 'RuntimeResponseGenericRuntimeResponseTypeAudio') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self,
               other: 'RuntimeResponseGenericRuntimeResponseTypeAudio') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RuntimeResponseGenericRuntimeResponseTypeChannelTransfer(
        RuntimeResponseGeneric):
    """
    RuntimeResponseGenericRuntimeResponseTypeChannelTransfer.

    :attr str response_type: The type of response returned by the dialog node. The
          specified response type must be supported by the client application or channel.
    :attr str message_to_user: The message to display to the user when initiating a
          channel transfer.
    :attr ChannelTransferInfo transfer_info: Information used by an integration to
          transfer the conversation to a different channel.
    :attr List[ResponseGenericChannel] channels: (optional) An array of objects
          specifying channels for which the response is intended. If **channels** is
          present, the response is intended for a built-in integration and should not be
          handled by an API client.
    """

    def __init__(self,
                 response_type: str,
                 message_to_user: str,
                 transfer_info: 'ChannelTransferInfo',
                 *,
                 channels: List['ResponseGenericChannel'] = None) -> None:
        """
        Initialize a RuntimeResponseGenericRuntimeResponseTypeChannelTransfer object.

        :param str response_type: The type of response returned by the dialog node.
               The specified response type must be supported by the client application or
               channel.
        :param str message_to_user: The message to display to the user when
               initiating a channel transfer.
        :param ChannelTransferInfo transfer_info: Information used by an
               integration to transfer the conversation to a different channel.
        :param List[ResponseGenericChannel] channels: (optional) An array of
               objects specifying channels for which the response is intended. If
               **channels** is present, the response is intended for a built-in
               integration and should not be handled by an API client.
        """
        # pylint: disable=super-init-not-called
        self.response_type = response_type
        self.message_to_user = message_to_user
        self.transfer_info = transfer_info
        self.channels = channels

    @classmethod
    def from_dict(
        cls, _dict: Dict
    ) -> 'RuntimeResponseGenericRuntimeResponseTypeChannelTransfer':
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeChannelTransfer object from a json dictionary."""
        args = {}
        if 'response_type' in _dict:
            args['response_type'] = _dict.get('response_type')
        else:
            raise ValueError(
                'Required property \'response_type\' not present in RuntimeResponseGenericRuntimeResponseTypeChannelTransfer JSON'
            )
        if 'message_to_user' in _dict:
            args['message_to_user'] = _dict.get('message_to_user')
        else:
            raise ValueError(
                'Required property \'message_to_user\' not present in RuntimeResponseGenericRuntimeResponseTypeChannelTransfer JSON'
            )
        if 'transfer_info' in _dict:
            args['transfer_info'] = ChannelTransferInfo.from_dict(
                _dict.get('transfer_info'))
        else:
            raise ValueError(
                'Required property \'transfer_info\' not present in RuntimeResponseGenericRuntimeResponseTypeChannelTransfer JSON'
            )
        if 'channels' in _dict:
            args['channels'] = [
                ResponseGenericChannel.from_dict(x)
                for x in _dict.get('channels')
            ]
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeChannelTransfer object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'response_type') and self.response_type is not None:
            _dict['response_type'] = self.response_type
        if hasattr(self,
                   'message_to_user') and self.message_to_user is not None:
            _dict['message_to_user'] = self.message_to_user
        if hasattr(self, 'transfer_info') and self.transfer_info is not None:
            _dict['transfer_info'] = self.transfer_info.to_dict()
        if hasattr(self, 'channels') and self.channels is not None:
            _dict['channels'] = [x.to_dict() for x in self.channels]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeResponseGenericRuntimeResponseTypeChannelTransfer object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(
        self, other: 'RuntimeResponseGenericRuntimeResponseTypeChannelTransfer'
    ) -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(
        self, other: 'RuntimeResponseGenericRuntimeResponseTypeChannelTransfer'
    ) -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RuntimeResponseGenericRuntimeResponseTypeConnectToAgent(
        RuntimeResponseGeneric):
    """
    RuntimeResponseGenericRuntimeResponseTypeConnectToAgent.

    :attr str response_type: The type of response returned by the dialog node. The
          specified response type must be supported by the client application or channel.
    :attr str message_to_human_agent: (optional) A message to be sent to the human
          agent who will be taking over the conversation.
    :attr AgentAvailabilityMessage agent_available: (optional) An optional message
          to be displayed to the user to indicate that the conversation will be
          transferred to the next available agent.
    :attr AgentAvailabilityMessage agent_unavailable: (optional) An optional message
          to be displayed to the user to indicate that no online agent is available to
          take over the conversation.
    :attr DialogNodeOutputConnectToAgentTransferInfo transfer_info: (optional)
          Routing or other contextual information to be used by target service desk
          systems.
    :attr str topic: (optional) A label identifying the topic of the conversation,
          derived from the **title** property of the relevant node or the **topic**
          property of the dialog node response.
    :attr List[ResponseGenericChannel] channels: (optional) An array of objects
          specifying channels for which the response is intended. If **channels** is
          present, the response is intended for a built-in integration and should not be
          handled by an API client.
    """

    def __init__(
            self,
            response_type: str,
            *,
            message_to_human_agent: str = None,
            agent_available: 'AgentAvailabilityMessage' = None,
            agent_unavailable: 'AgentAvailabilityMessage' = None,
            transfer_info: 'DialogNodeOutputConnectToAgentTransferInfo' = None,
            topic: str = None,
            channels: List['ResponseGenericChannel'] = None) -> None:
        """
        Initialize a RuntimeResponseGenericRuntimeResponseTypeConnectToAgent object.

        :param str response_type: The type of response returned by the dialog node.
               The specified response type must be supported by the client application or
               channel.
        :param str message_to_human_agent: (optional) A message to be sent to the
               human agent who will be taking over the conversation.
        :param AgentAvailabilityMessage agent_available: (optional) An optional
               message to be displayed to the user to indicate that the conversation will
               be transferred to the next available agent.
        :param AgentAvailabilityMessage agent_unavailable: (optional) An optional
               message to be displayed to the user to indicate that no online agent is
               available to take over the conversation.
        :param DialogNodeOutputConnectToAgentTransferInfo transfer_info: (optional)
               Routing or other contextual information to be used by target service desk
               systems.
        :param str topic: (optional) A label identifying the topic of the
               conversation, derived from the **title** property of the relevant node or
               the **topic** property of the dialog node response.
        :param List[ResponseGenericChannel] channels: (optional) An array of
               objects specifying channels for which the response is intended. If
               **channels** is present, the response is intended for a built-in
               integration and should not be handled by an API client.
        """
        # pylint: disable=super-init-not-called
        self.response_type = response_type
        self.message_to_human_agent = message_to_human_agent
        self.agent_available = agent_available
        self.agent_unavailable = agent_unavailable
        self.transfer_info = transfer_info
        self.topic = topic
        self.channels = channels

    @classmethod
    def from_dict(
        cls, _dict: Dict
    ) -> 'RuntimeResponseGenericRuntimeResponseTypeConnectToAgent':
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeConnectToAgent object from a json dictionary."""
        args = {}
        if 'response_type' in _dict:
            args['response_type'] = _dict.get('response_type')
        else:
            raise ValueError(
                'Required property \'response_type\' not present in RuntimeResponseGenericRuntimeResponseTypeConnectToAgent JSON'
            )
        if 'message_to_human_agent' in _dict:
            args['message_to_human_agent'] = _dict.get('message_to_human_agent')
        if 'agent_available' in _dict:
            args['agent_available'] = AgentAvailabilityMessage.from_dict(
                _dict.get('agent_available'))
        if 'agent_unavailable' in _dict:
            args['agent_unavailable'] = AgentAvailabilityMessage.from_dict(
                _dict.get('agent_unavailable'))
        if 'transfer_info' in _dict:
            args[
                'transfer_info'] = DialogNodeOutputConnectToAgentTransferInfo.from_dict(
                    _dict.get('transfer_info'))
        if 'topic' in _dict:
            args['topic'] = _dict.get('topic')
        if 'channels' in _dict:
            args['channels'] = [
                ResponseGenericChannel.from_dict(x)
                for x in _dict.get('channels')
            ]
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeConnectToAgent object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'response_type') and self.response_type is not None:
            _dict['response_type'] = self.response_type
        if hasattr(self, 'message_to_human_agent'
                  ) and self.message_to_human_agent is not None:
            _dict['message_to_human_agent'] = self.message_to_human_agent
        if hasattr(self,
                   'agent_available') and self.agent_available is not None:
            _dict['agent_available'] = self.agent_available.to_dict()
        if hasattr(self,
                   'agent_unavailable') and self.agent_unavailable is not None:
            _dict['agent_unavailable'] = self.agent_unavailable.to_dict()
        if hasattr(self, 'transfer_info') and self.transfer_info is not None:
            _dict['transfer_info'] = self.transfer_info.to_dict()
        if hasattr(self, 'topic') and self.topic is not None:
            _dict['topic'] = self.topic
        if hasattr(self, 'channels') and self.channels is not None:
            _dict['channels'] = [x.to_dict() for x in self.channels]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeResponseGenericRuntimeResponseTypeConnectToAgent object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(
        self, other: 'RuntimeResponseGenericRuntimeResponseTypeConnectToAgent'
    ) -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(
        self, other: 'RuntimeResponseGenericRuntimeResponseTypeConnectToAgent'
    ) -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RuntimeResponseGenericRuntimeResponseTypeIframe(RuntimeResponseGeneric):
    """
    RuntimeResponseGenericRuntimeResponseTypeIframe.

    :attr str response_type: The type of response returned by the dialog node. The
          specified response type must be supported by the client application or channel.
    :attr str source: The `https:` URL of the embeddable content.
    :attr str title: (optional) The title or introductory text to show before the
          response.
    :attr str description: (optional) The description to show with the the response.
    :attr str image_url: (optional) The URL of an image that shows a preview of the
          embedded content.
    :attr List[ResponseGenericChannel] channels: (optional) An array of objects
          specifying channels for which the response is intended. If **channels** is
          present, the response is intended for a built-in integration and should not be
          handled by an API client.
    """

    def __init__(self,
                 response_type: str,
                 source: str,
                 *,
                 title: str = None,
                 description: str = None,
                 image_url: str = None,
                 channels: List['ResponseGenericChannel'] = None) -> None:
        """
        Initialize a RuntimeResponseGenericRuntimeResponseTypeIframe object.

        :param str response_type: The type of response returned by the dialog node.
               The specified response type must be supported by the client application or
               channel.
        :param str source: The `https:` URL of the embeddable content.
        :param str title: (optional) The title or introductory text to show before
               the response.
        :param str description: (optional) The description to show with the the
               response.
        :param str image_url: (optional) The URL of an image that shows a preview
               of the embedded content.
        :param List[ResponseGenericChannel] channels: (optional) An array of
               objects specifying channels for which the response is intended. If
               **channels** is present, the response is intended for a built-in
               integration and should not be handled by an API client.
        """
        # pylint: disable=super-init-not-called
        self.response_type = response_type
        self.source = source
        self.title = title
        self.description = description
        self.image_url = image_url
        self.channels = channels

    @classmethod
    def from_dict(
            cls,
            _dict: Dict) -> 'RuntimeResponseGenericRuntimeResponseTypeIframe':
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeIframe object from a json dictionary."""
        args = {}
        if 'response_type' in _dict:
            args['response_type'] = _dict.get('response_type')
        else:
            raise ValueError(
                'Required property \'response_type\' not present in RuntimeResponseGenericRuntimeResponseTypeIframe JSON'
            )
        if 'source' in _dict:
            args['source'] = _dict.get('source')
        else:
            raise ValueError(
                'Required property \'source\' not present in RuntimeResponseGenericRuntimeResponseTypeIframe JSON'
            )
        if 'title' in _dict:
            args['title'] = _dict.get('title')
        if 'description' in _dict:
            args['description'] = _dict.get('description')
        if 'image_url' in _dict:
            args['image_url'] = _dict.get('image_url')
        if 'channels' in _dict:
            args['channels'] = [
                ResponseGenericChannel.from_dict(x)
                for x in _dict.get('channels')
            ]
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeIframe object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'response_type') and self.response_type is not None:
            _dict['response_type'] = self.response_type
        if hasattr(self, 'source') and self.source is not None:
            _dict['source'] = self.source
        if hasattr(self, 'title') and self.title is not None:
            _dict['title'] = self.title
        if hasattr(self, 'description') and self.description is not None:
            _dict['description'] = self.description
        if hasattr(self, 'image_url') and self.image_url is not None:
            _dict['image_url'] = self.image_url
        if hasattr(self, 'channels') and self.channels is not None:
            _dict['channels'] = [x.to_dict() for x in self.channels]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeResponseGenericRuntimeResponseTypeIframe object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(
            self,
            other: 'RuntimeResponseGenericRuntimeResponseTypeIframe') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(
            self,
            other: 'RuntimeResponseGenericRuntimeResponseTypeIframe') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RuntimeResponseGenericRuntimeResponseTypeImage(RuntimeResponseGeneric):
    """
    RuntimeResponseGenericRuntimeResponseTypeImage.

    :attr str response_type: The type of response returned by the dialog node. The
          specified response type must be supported by the client application or channel.
    :attr str source: The `https:` URL of the image.
    :attr str title: (optional) The title to show before the response.
    :attr str description: (optional) The description to show with the the response.
    :attr List[ResponseGenericChannel] channels: (optional) An array of objects
          specifying channels for which the response is intended. If **channels** is
          present, the response is intended for a built-in integration and should not be
          handled by an API client.
    :attr str alt_text: (optional) Descriptive text that can be used for screen
          readers or other situations where the image cannot be seen.
    """

    def __init__(self,
                 response_type: str,
                 source: str,
                 *,
                 title: str = None,
                 description: str = None,
                 channels: List['ResponseGenericChannel'] = None,
                 alt_text: str = None) -> None:
        """
        Initialize a RuntimeResponseGenericRuntimeResponseTypeImage object.

        :param str response_type: The type of response returned by the dialog node.
               The specified response type must be supported by the client application or
               channel.
        :param str source: The `https:` URL of the image.
        :param str title: (optional) The title to show before the response.
        :param str description: (optional) The description to show with the the
               response.
        :param List[ResponseGenericChannel] channels: (optional) An array of
               objects specifying channels for which the response is intended. If
               **channels** is present, the response is intended for a built-in
               integration and should not be handled by an API client.
        :param str alt_text: (optional) Descriptive text that can be used for
               screen readers or other situations where the image cannot be seen.
        """
        # pylint: disable=super-init-not-called
        self.response_type = response_type
        self.source = source
        self.title = title
        self.description = description
        self.channels = channels
        self.alt_text = alt_text

    @classmethod
    def from_dict(
            cls,
            _dict: Dict) -> 'RuntimeResponseGenericRuntimeResponseTypeImage':
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeImage object from a json dictionary."""
        args = {}
        if 'response_type' in _dict:
            args['response_type'] = _dict.get('response_type')
        else:
            raise ValueError(
                'Required property \'response_type\' not present in RuntimeResponseGenericRuntimeResponseTypeImage JSON'
            )
        if 'source' in _dict:
            args['source'] = _dict.get('source')
        else:
            raise ValueError(
                'Required property \'source\' not present in RuntimeResponseGenericRuntimeResponseTypeImage JSON'
            )
        if 'title' in _dict:
            args['title'] = _dict.get('title')
        if 'description' in _dict:
            args['description'] = _dict.get('description')
        if 'channels' in _dict:
            args['channels'] = [
                ResponseGenericChannel.from_dict(x)
                for x in _dict.get('channels')
            ]
        if 'alt_text' in _dict:
            args['alt_text'] = _dict.get('alt_text')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeImage object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'response_type') and self.response_type is not None:
            _dict['response_type'] = self.response_type
        if hasattr(self, 'source') and self.source is not None:
            _dict['source'] = self.source
        if hasattr(self, 'title') and self.title is not None:
            _dict['title'] = self.title
        if hasattr(self, 'description') and self.description is not None:
            _dict['description'] = self.description
        if hasattr(self, 'channels') and self.channels is not None:
            _dict['channels'] = [x.to_dict() for x in self.channels]
        if hasattr(self, 'alt_text') and self.alt_text is not None:
            _dict['alt_text'] = self.alt_text
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeResponseGenericRuntimeResponseTypeImage object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self,
               other: 'RuntimeResponseGenericRuntimeResponseTypeImage') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self,
               other: 'RuntimeResponseGenericRuntimeResponseTypeImage') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RuntimeResponseGenericRuntimeResponseTypeOption(RuntimeResponseGeneric):
    """
    RuntimeResponseGenericRuntimeResponseTypeOption.

    :attr str response_type: The type of response returned by the dialog node. The
          specified response type must be supported by the client application or channel.
    :attr str title: The title or introductory text to show before the response.
    :attr str description: (optional) The description to show with the the response.
    :attr str preference: (optional) The preferred type of control to display.
    :attr List[DialogNodeOutputOptionsElement] options: An array of objects
          describing the options from which the user can choose.
    :attr List[ResponseGenericChannel] channels: (optional) An array of objects
          specifying channels for which the response is intended. If **channels** is
          present, the response is intended for a built-in integration and should not be
          handled by an API client.
    """

    def __init__(self,
                 response_type: str,
                 title: str,
                 options: List['DialogNodeOutputOptionsElement'],
                 *,
                 description: str = None,
                 preference: str = None,
                 channels: List['ResponseGenericChannel'] = None) -> None:
        """
        Initialize a RuntimeResponseGenericRuntimeResponseTypeOption object.

        :param str response_type: The type of response returned by the dialog node.
               The specified response type must be supported by the client application or
               channel.
        :param str title: The title or introductory text to show before the
               response.
        :param List[DialogNodeOutputOptionsElement] options: An array of objects
               describing the options from which the user can choose.
        :param str description: (optional) The description to show with the the
               response.
        :param str preference: (optional) The preferred type of control to display.
        :param List[ResponseGenericChannel] channels: (optional) An array of
               objects specifying channels for which the response is intended. If
               **channels** is present, the response is intended for a built-in
               integration and should not be handled by an API client.
        """
        # pylint: disable=super-init-not-called
        self.response_type = response_type
        self.title = title
        self.description = description
        self.preference = preference
        self.options = options
        self.channels = channels

    @classmethod
    def from_dict(
            cls,
            _dict: Dict) -> 'RuntimeResponseGenericRuntimeResponseTypeOption':
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeOption object from a json dictionary."""
        args = {}
        if 'response_type' in _dict:
            args['response_type'] = _dict.get('response_type')
        else:
            raise ValueError(
                'Required property \'response_type\' not present in RuntimeResponseGenericRuntimeResponseTypeOption JSON'
            )
        if 'title' in _dict:
            args['title'] = _dict.get('title')
        else:
            raise ValueError(
                'Required property \'title\' not present in RuntimeResponseGenericRuntimeResponseTypeOption JSON'
            )
        if 'description' in _dict:
            args['description'] = _dict.get('description')
        if 'preference' in _dict:
            args['preference'] = _dict.get('preference')
        if 'options' in _dict:
            args['options'] = [
                DialogNodeOutputOptionsElement.from_dict(x)
                for x in _dict.get('options')
            ]
        else:
            raise ValueError(
                'Required property \'options\' not present in RuntimeResponseGenericRuntimeResponseTypeOption JSON'
            )
        if 'channels' in _dict:
            args['channels'] = [
                ResponseGenericChannel.from_dict(x)
                for x in _dict.get('channels')
            ]
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeOption object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'response_type') and self.response_type is not None:
            _dict['response_type'] = self.response_type
        if hasattr(self, 'title') and self.title is not None:
            _dict['title'] = self.title
        if hasattr(self, 'description') and self.description is not None:
            _dict['description'] = self.description
        if hasattr(self, 'preference') and self.preference is not None:
            _dict['preference'] = self.preference
        if hasattr(self, 'options') and self.options is not None:
            _dict['options'] = [x.to_dict() for x in self.options]
        if hasattr(self, 'channels') and self.channels is not None:
            _dict['channels'] = [x.to_dict() for x in self.channels]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeResponseGenericRuntimeResponseTypeOption object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(
            self,
            other: 'RuntimeResponseGenericRuntimeResponseTypeOption') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(
            self,
            other: 'RuntimeResponseGenericRuntimeResponseTypeOption') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class PreferenceEnum(str, Enum):
        """
        The preferred type of control to display.
        """
        DROPDOWN = 'dropdown'
        BUTTON = 'button'


class RuntimeResponseGenericRuntimeResponseTypePause(RuntimeResponseGeneric):
    """
    RuntimeResponseGenericRuntimeResponseTypePause.

    :attr str response_type: The type of response returned by the dialog node. The
          specified response type must be supported by the client application or channel.
    :attr int time: How long to pause, in milliseconds.
    :attr bool typing: (optional) Whether to send a "user is typing" event during
          the pause.
    :attr List[ResponseGenericChannel] channels: (optional) An array of objects
          specifying channels for which the response is intended. If **channels** is
          present, the response is intended for a built-in integration and should not be
          handled by an API client.
    """

    def __init__(self,
                 response_type: str,
                 time: int,
                 *,
                 typing: bool = None,
                 channels: List['ResponseGenericChannel'] = None) -> None:
        """
        Initialize a RuntimeResponseGenericRuntimeResponseTypePause object.

        :param str response_type: The type of response returned by the dialog node.
               The specified response type must be supported by the client application or
               channel.
        :param int time: How long to pause, in milliseconds.
        :param bool typing: (optional) Whether to send a "user is typing" event
               during the pause.
        :param List[ResponseGenericChannel] channels: (optional) An array of
               objects specifying channels for which the response is intended. If
               **channels** is present, the response is intended for a built-in
               integration and should not be handled by an API client.
        """
        # pylint: disable=super-init-not-called
        self.response_type = response_type
        self.time = time
        self.typing = typing
        self.channels = channels

    @classmethod
    def from_dict(
            cls,
            _dict: Dict) -> 'RuntimeResponseGenericRuntimeResponseTypePause':
        """Initialize a RuntimeResponseGenericRuntimeResponseTypePause object from a json dictionary."""
        args = {}
        if 'response_type' in _dict:
            args['response_type'] = _dict.get('response_type')
        else:
            raise ValueError(
                'Required property \'response_type\' not present in RuntimeResponseGenericRuntimeResponseTypePause JSON'
            )
        if 'time' in _dict:
            args['time'] = _dict.get('time')
        else:
            raise ValueError(
                'Required property \'time\' not present in RuntimeResponseGenericRuntimeResponseTypePause JSON'
            )
        if 'typing' in _dict:
            args['typing'] = _dict.get('typing')
        if 'channels' in _dict:
            args['channels'] = [
                ResponseGenericChannel.from_dict(x)
                for x in _dict.get('channels')
            ]
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeResponseGenericRuntimeResponseTypePause object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'response_type') and self.response_type is not None:
            _dict['response_type'] = self.response_type
        if hasattr(self, 'time') and self.time is not None:
            _dict['time'] = self.time
        if hasattr(self, 'typing') and self.typing is not None:
            _dict['typing'] = self.typing
        if hasattr(self, 'channels') and self.channels is not None:
            _dict['channels'] = [x.to_dict() for x in self.channels]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeResponseGenericRuntimeResponseTypePause object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self,
               other: 'RuntimeResponseGenericRuntimeResponseTypePause') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self,
               other: 'RuntimeResponseGenericRuntimeResponseTypePause') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RuntimeResponseGenericRuntimeResponseTypeSearch(RuntimeResponseGeneric):
    """
    RuntimeResponseGenericRuntimeResponseTypeSearch.

    :attr str response_type: The type of response returned by the dialog node. The
          specified response type must be supported by the client application or channel.
    :attr str header: The title or introductory text to show before the response.
          This text is defined in the search skill configuration.
    :attr List[SearchResult] primary_results: An array of objects that contains the
          search results to be displayed in the initial response to the user.
    :attr List[SearchResult] additional_results: An array of objects that contains
          additional search results that can be displayed to the user upon request.
    :attr List[ResponseGenericChannel] channels: (optional) An array of objects
          specifying channels for which the response is intended. If **channels** is
          present, the response is intended for a built-in integration and should not be
          handled by an API client.
    """

    def __init__(self,
                 response_type: str,
                 header: str,
                 primary_results: List['SearchResult'],
                 additional_results: List['SearchResult'],
                 *,
                 channels: List['ResponseGenericChannel'] = None) -> None:
        """
        Initialize a RuntimeResponseGenericRuntimeResponseTypeSearch object.

        :param str response_type: The type of response returned by the dialog node.
               The specified response type must be supported by the client application or
               channel.
        :param str header: The title or introductory text to show before the
               response. This text is defined in the search skill configuration.
        :param List[SearchResult] primary_results: An array of objects that
               contains the search results to be displayed in the initial response to the
               user.
        :param List[SearchResult] additional_results: An array of objects that
               contains additional search results that can be displayed to the user upon
               request.
        :param List[ResponseGenericChannel] channels: (optional) An array of
               objects specifying channels for which the response is intended. If
               **channels** is present, the response is intended for a built-in
               integration and should not be handled by an API client.
        """
        # pylint: disable=super-init-not-called
        self.response_type = response_type
        self.header = header
        self.primary_results = primary_results
        self.additional_results = additional_results
        self.channels = channels

    @classmethod
    def from_dict(
            cls,
            _dict: Dict) -> 'RuntimeResponseGenericRuntimeResponseTypeSearch':
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeSearch object from a json dictionary."""
        args = {}
        if 'response_type' in _dict:
            args['response_type'] = _dict.get('response_type')
        else:
            raise ValueError(
                'Required property \'response_type\' not present in RuntimeResponseGenericRuntimeResponseTypeSearch JSON'
            )
        if 'header' in _dict:
            args['header'] = _dict.get('header')
        else:
            raise ValueError(
                'Required property \'header\' not present in RuntimeResponseGenericRuntimeResponseTypeSearch JSON'
            )
        if 'primary_results' in _dict:
            args['primary_results'] = [
                SearchResult.from_dict(x) for x in _dict.get('primary_results')
            ]
        else:
            raise ValueError(
                'Required property \'primary_results\' not present in RuntimeResponseGenericRuntimeResponseTypeSearch JSON'
            )
        if 'additional_results' in _dict:
            args['additional_results'] = [
                SearchResult.from_dict(x)
                for x in _dict.get('additional_results')
            ]
        else:
            raise ValueError(
                'Required property \'additional_results\' not present in RuntimeResponseGenericRuntimeResponseTypeSearch JSON'
            )
        if 'channels' in _dict:
            args['channels'] = [
                ResponseGenericChannel.from_dict(x)
                for x in _dict.get('channels')
            ]
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeSearch object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'response_type') and self.response_type is not None:
            _dict['response_type'] = self.response_type
        if hasattr(self, 'header') and self.header is not None:
            _dict['header'] = self.header
        if hasattr(self,
                   'primary_results') and self.primary_results is not None:
            _dict['primary_results'] = [
                x.to_dict() for x in self.primary_results
            ]
        if hasattr(
                self,
                'additional_results') and self.additional_results is not None:
            _dict['additional_results'] = [
                x.to_dict() for x in self.additional_results
            ]
        if hasattr(self, 'channels') and self.channels is not None:
            _dict['channels'] = [x.to_dict() for x in self.channels]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeResponseGenericRuntimeResponseTypeSearch object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(
            self,
            other: 'RuntimeResponseGenericRuntimeResponseTypeSearch') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(
            self,
            other: 'RuntimeResponseGenericRuntimeResponseTypeSearch') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RuntimeResponseGenericRuntimeResponseTypeSuggestion(
        RuntimeResponseGeneric):
    """
    RuntimeResponseGenericRuntimeResponseTypeSuggestion.

    :attr str response_type: The type of response returned by the dialog node. The
          specified response type must be supported by the client application or channel.
    :attr str title: The title or introductory text to show before the response.
    :attr List[DialogSuggestion] suggestions: An array of objects describing the
          possible matching dialog nodes from which the user can choose.
    :attr List[ResponseGenericChannel] channels: (optional) An array of objects
          specifying channels for which the response is intended. If **channels** is
          present, the response is intended for a built-in integration and should not be
          handled by an API client.
    """

    def __init__(self,
                 response_type: str,
                 title: str,
                 suggestions: List['DialogSuggestion'],
                 *,
                 channels: List['ResponseGenericChannel'] = None) -> None:
        """
        Initialize a RuntimeResponseGenericRuntimeResponseTypeSuggestion object.

        :param str response_type: The type of response returned by the dialog node.
               The specified response type must be supported by the client application or
               channel.
        :param str title: The title or introductory text to show before the
               response.
        :param List[DialogSuggestion] suggestions: An array of objects describing
               the possible matching dialog nodes from which the user can choose.
        :param List[ResponseGenericChannel] channels: (optional) An array of
               objects specifying channels for which the response is intended. If
               **channels** is present, the response is intended for a built-in
               integration and should not be handled by an API client.
        """
        # pylint: disable=super-init-not-called
        self.response_type = response_type
        self.title = title
        self.suggestions = suggestions
        self.channels = channels

    @classmethod
    def from_dict(
            cls, _dict: Dict
    ) -> 'RuntimeResponseGenericRuntimeResponseTypeSuggestion':
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeSuggestion object from a json dictionary."""
        args = {}
        if 'response_type' in _dict:
            args['response_type'] = _dict.get('response_type')
        else:
            raise ValueError(
                'Required property \'response_type\' not present in RuntimeResponseGenericRuntimeResponseTypeSuggestion JSON'
            )
        if 'title' in _dict:
            args['title'] = _dict.get('title')
        else:
            raise ValueError(
                'Required property \'title\' not present in RuntimeResponseGenericRuntimeResponseTypeSuggestion JSON'
            )
        if 'suggestions' in _dict:
            args['suggestions'] = [
                DialogSuggestion.from_dict(x) for x in _dict.get('suggestions')
            ]
        else:
            raise ValueError(
                'Required property \'suggestions\' not present in RuntimeResponseGenericRuntimeResponseTypeSuggestion JSON'
            )
        if 'channels' in _dict:
            args['channels'] = [
                ResponseGenericChannel.from_dict(x)
                for x in _dict.get('channels')
            ]
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeSuggestion object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'response_type') and self.response_type is not None:
            _dict['response_type'] = self.response_type
        if hasattr(self, 'title') and self.title is not None:
            _dict['title'] = self.title
        if hasattr(self, 'suggestions') and self.suggestions is not None:
            _dict['suggestions'] = [x.to_dict() for x in self.suggestions]
        if hasattr(self, 'channels') and self.channels is not None:
            _dict['channels'] = [x.to_dict() for x in self.channels]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeResponseGenericRuntimeResponseTypeSuggestion object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(
            self, other: 'RuntimeResponseGenericRuntimeResponseTypeSuggestion'
    ) -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(
            self, other: 'RuntimeResponseGenericRuntimeResponseTypeSuggestion'
    ) -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RuntimeResponseGenericRuntimeResponseTypeText(RuntimeResponseGeneric):
    """
    RuntimeResponseGenericRuntimeResponseTypeText.

    :attr str response_type: The type of response returned by the dialog node. The
          specified response type must be supported by the client application or channel.
    :attr str text: The text of the response.
    :attr List[ResponseGenericChannel] channels: (optional) An array of objects
          specifying channels for which the response is intended. If **channels** is
          present, the response is intended for a built-in integration and should not be
          handled by an API client.
    """

    def __init__(self,
                 response_type: str,
                 text: str,
                 *,
                 channels: List['ResponseGenericChannel'] = None) -> None:
        """
        Initialize a RuntimeResponseGenericRuntimeResponseTypeText object.

        :param str response_type: The type of response returned by the dialog node.
               The specified response type must be supported by the client application or
               channel.
        :param str text: The text of the response.
        :param List[ResponseGenericChannel] channels: (optional) An array of
               objects specifying channels for which the response is intended. If
               **channels** is present, the response is intended for a built-in
               integration and should not be handled by an API client.
        """
        # pylint: disable=super-init-not-called
        self.response_type = response_type
        self.text = text
        self.channels = channels

    @classmethod
    def from_dict(
            cls,
            _dict: Dict) -> 'RuntimeResponseGenericRuntimeResponseTypeText':
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeText object from a json dictionary."""
        args = {}
        if 'response_type' in _dict:
            args['response_type'] = _dict.get('response_type')
        else:
            raise ValueError(
                'Required property \'response_type\' not present in RuntimeResponseGenericRuntimeResponseTypeText JSON'
            )
        if 'text' in _dict:
            args['text'] = _dict.get('text')
        else:
            raise ValueError(
                'Required property \'text\' not present in RuntimeResponseGenericRuntimeResponseTypeText JSON'
            )
        if 'channels' in _dict:
            args['channels'] = [
                ResponseGenericChannel.from_dict(x)
                for x in _dict.get('channels')
            ]
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeText object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'response_type') and self.response_type is not None:
            _dict['response_type'] = self.response_type
        if hasattr(self, 'text') and self.text is not None:
            _dict['text'] = self.text
        if hasattr(self, 'channels') and self.channels is not None:
            _dict['channels'] = [x.to_dict() for x in self.channels]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeResponseGenericRuntimeResponseTypeText object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self,
               other: 'RuntimeResponseGenericRuntimeResponseTypeText') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self,
               other: 'RuntimeResponseGenericRuntimeResponseTypeText') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RuntimeResponseGenericRuntimeResponseTypeUserDefined(
        RuntimeResponseGeneric):
    """
    RuntimeResponseGenericRuntimeResponseTypeUserDefined.

    :attr str response_type: The type of response returned by the dialog node. The
          specified response type must be supported by the client application or channel.
    :attr dict user_defined: An object containing any properties for the
          user-defined response type.
    :attr List[ResponseGenericChannel] channels: (optional) An array of objects
          specifying channels for which the response is intended. If **channels** is
          present, the response is intended for a built-in integration and should not be
          handled by an API client.
    """

    def __init__(self,
                 response_type: str,
                 user_defined: dict,
                 *,
                 channels: List['ResponseGenericChannel'] = None) -> None:
        """
        Initialize a RuntimeResponseGenericRuntimeResponseTypeUserDefined object.

        :param str response_type: The type of response returned by the dialog node.
               The specified response type must be supported by the client application or
               channel.
        :param dict user_defined: An object containing any properties for the
               user-defined response type.
        :param List[ResponseGenericChannel] channels: (optional) An array of
               objects specifying channels for which the response is intended. If
               **channels** is present, the response is intended for a built-in
               integration and should not be handled by an API client.
        """
        # pylint: disable=super-init-not-called
        self.response_type = response_type
        self.user_defined = user_defined
        self.channels = channels

    @classmethod
    def from_dict(
            cls, _dict: Dict
    ) -> 'RuntimeResponseGenericRuntimeResponseTypeUserDefined':
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeUserDefined object from a json dictionary."""
        args = {}
        if 'response_type' in _dict:
            args['response_type'] = _dict.get('response_type')
        else:
            raise ValueError(
                'Required property \'response_type\' not present in RuntimeResponseGenericRuntimeResponseTypeUserDefined JSON'
            )
        if 'user_defined' in _dict:
            args['user_defined'] = _dict.get('user_defined')
        else:
            raise ValueError(
                'Required property \'user_defined\' not present in RuntimeResponseGenericRuntimeResponseTypeUserDefined JSON'
            )
        if 'channels' in _dict:
            args['channels'] = [
                ResponseGenericChannel.from_dict(x)
                for x in _dict.get('channels')
            ]
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeUserDefined object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'response_type') and self.response_type is not None:
            _dict['response_type'] = self.response_type
        if hasattr(self, 'user_defined') and self.user_defined is not None:
            _dict['user_defined'] = self.user_defined
        if hasattr(self, 'channels') and self.channels is not None:
            _dict['channels'] = [x.to_dict() for x in self.channels]
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeResponseGenericRuntimeResponseTypeUserDefined object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(
            self, other: 'RuntimeResponseGenericRuntimeResponseTypeUserDefined'
    ) -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(
            self, other: 'RuntimeResponseGenericRuntimeResponseTypeUserDefined'
    ) -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other


class RuntimeResponseGenericRuntimeResponseTypeVideo(RuntimeResponseGeneric):
    """
    RuntimeResponseGenericRuntimeResponseTypeVideo.

    :attr str response_type: The type of response returned by the dialog node. The
          specified response type must be supported by the client application or channel.
    :attr str source: The `https:` URL of the video.
    :attr str title: (optional) The title or introductory text to show before the
          response.
    :attr str description: (optional) The description to show with the the response.
    :attr List[ResponseGenericChannel] channels: (optional) An array of objects
          specifying channels for which the response is intended. If **channels** is
          present, the response is intended for a built-in integration and should not be
          handled by an API client.
    :attr object channel_options: (optional) For internal use only.
    :attr str alt_text: (optional) Descriptive text that can be used for screen
          readers or other situations where the video cannot be seen.
    """

    def __init__(self,
                 response_type: str,
                 source: str,
                 *,
                 title: str = None,
                 description: str = None,
                 channels: List['ResponseGenericChannel'] = None,
                 channel_options: object = None,
                 alt_text: str = None) -> None:
        """
        Initialize a RuntimeResponseGenericRuntimeResponseTypeVideo object.

        :param str response_type: The type of response returned by the dialog node.
               The specified response type must be supported by the client application or
               channel.
        :param str source: The `https:` URL of the video.
        :param str title: (optional) The title or introductory text to show before
               the response.
        :param str description: (optional) The description to show with the the
               response.
        :param List[ResponseGenericChannel] channels: (optional) An array of
               objects specifying channels for which the response is intended. If
               **channels** is present, the response is intended for a built-in
               integration and should not be handled by an API client.
        :param object channel_options: (optional) For internal use only.
        :param str alt_text: (optional) Descriptive text that can be used for
               screen readers or other situations where the video cannot be seen.
        """
        # pylint: disable=super-init-not-called
        self.response_type = response_type
        self.source = source
        self.title = title
        self.description = description
        self.channels = channels
        self.channel_options = channel_options
        self.alt_text = alt_text

    @classmethod
    def from_dict(
            cls,
            _dict: Dict) -> 'RuntimeResponseGenericRuntimeResponseTypeVideo':
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeVideo object from a json dictionary."""
        args = {}
        if 'response_type' in _dict:
            args['response_type'] = _dict.get('response_type')
        else:
            raise ValueError(
                'Required property \'response_type\' not present in RuntimeResponseGenericRuntimeResponseTypeVideo JSON'
            )
        if 'source' in _dict:
            args['source'] = _dict.get('source')
        else:
            raise ValueError(
                'Required property \'source\' not present in RuntimeResponseGenericRuntimeResponseTypeVideo JSON'
            )
        if 'title' in _dict:
            args['title'] = _dict.get('title')
        if 'description' in _dict:
            args['description'] = _dict.get('description')
        if 'channels' in _dict:
            args['channels'] = [
                ResponseGenericChannel.from_dict(x)
                for x in _dict.get('channels')
            ]
        if 'channel_options' in _dict:
            args['channel_options'] = _dict.get('channel_options')
        if 'alt_text' in _dict:
            args['alt_text'] = _dict.get('alt_text')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a RuntimeResponseGenericRuntimeResponseTypeVideo object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'response_type') and self.response_type is not None:
            _dict['response_type'] = self.response_type
        if hasattr(self, 'source') and self.source is not None:
            _dict['source'] = self.source
        if hasattr(self, 'title') and self.title is not None:
            _dict['title'] = self.title
        if hasattr(self, 'description') and self.description is not None:
            _dict['description'] = self.description
        if hasattr(self, 'channels') and self.channels is not None:
            _dict['channels'] = [x.to_dict() for x in self.channels]
        if hasattr(self,
                   'channel_options') and self.channel_options is not None:
            _dict['channel_options'] = self.channel_options
        if hasattr(self, 'alt_text') and self.alt_text is not None:
            _dict['alt_text'] = self.alt_text
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this RuntimeResponseGenericRuntimeResponseTypeVideo object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self,
               other: 'RuntimeResponseGenericRuntimeResponseTypeVideo') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self,
               other: 'RuntimeResponseGenericRuntimeResponseTypeVideo') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other
