/*	strip_DEBUG

PIRL CVS ID:	strip_DEBUG.java,v 1.13 2012/04/16 06:15:36 castalia Exp

Copyright (C) 2002-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Strings;

import	java.lang.String;
import	java.io.InputStream;
import	java.io.FileInputStream;
import	java.io.InputStreamReader;
import	java.io.FileNotFoundException;
import	java.io.StringWriter;
import	java.io.IOException;

/**	<I>strip_DEBUG</I> is a filter utility to remove "DEBUG" sections from
	source code files.
<P>
	Use:
<BLOCKQUOTE><PRE>
	<B>java strip_DEBUG</B> [&lt;<I>source filename</I>&gt;]
</PRE></BLOCKQUOTE>
<P>
	Since Java does not provide preprocessor capabilities, debugging code
	sections can not be surrounded by #ifdef statements as they are in
	C/C++. For Java, then, the in-line code debugging technique assumed
	here is the use of a "DEBUG" marker to identify sequences of code
	that are only used for debugging purposes. This utility strips
	"DEBUG" sequences from a source code file.
<p>
	A "DEBUG" sequence is identified by an expression containing the
	"DEBUG" marker. All source code lines containing the expression are
	stripped from the resultant output. The expression may be may be a
	variable or variable list which includes all members of the list, a
	method which includes the entire method definition, a conditional
	which includes the expression or expression block following the
	conditional, or any other expression. When expressions are stripped
	the entirertiy of all lines containing the expression are removed;
	thus "DEBUG" expressions must not be mixed with other expressions on
	the same line(s).
<P>
	<b>N.B.</b>: The "DEBUG" string that is immediately preceded by a
	whitespace character, end of line sequence or left parenthesis
	character is taken to be contained in a "DEBUG" expression. Thus
	comments containing the unquoted word "DEBUG" will be mistaken for
	part of an expression resulting in incorrectly stripping the line on
	which "DEBUG" occurs and the expression following the comment (unless
	the commend contains characters that can misinterpreted as the end of
	an expression). The "DEBUG"
<P>
	The sripped file is written to stdout. If no source filename is
	specified, it is read from stdin.
<P>
@author		Bradford Castalia, UA/PIRL
@version	1.13 
*/
public class strip_DEBUG
{
public static final String
	ID = "PIRL.Strings.strip_DEBUG (1.13 2012/04/16 06:15:36)";

private static StringBuffer
	Source;

private static final String
	WHITESPACE	= " \t\n\r",
	EOL			= "\n\r";

private static final int
	DEBUG_OFF		= 0,
	DEBUG_STRIP		= 1 << 0,
	DEBUG_BEGIN		= 1 << 1,
	DEBUG_END		= 1 << 2,
	DEBUG_ALL		= -1,

	DEBUG			= DEBUG_OFF;


public static void main (String[] arguments) 
{
//	Read in the entire file into the Source.
InputStreamReader
	source = null;
if (arguments.length == 0)
	source = new InputStreamReader (System.in);
else
	{
	try {source = new InputStreamReader (new FileInputStream (arguments[0]));}
	catch (FileNotFoundException exception)
		{
		System.err.println
			("strip_DEBUG: Unable to access file " + arguments[0] + '\n'
			 + exception.getMessage ());
		System.exit (1);
		}
	}
try
	{
	StringWriter
		source_writer = new StringWriter ();
	char
		characters[] = new char[1024];
	int
		count;
	while ((count = source.read (characters, 0, 1024)) != -1)
		   source_writer.write (characters, 0, count);
	Source = source_writer.getBuffer ();
	}
catch (IOException exception)
	{
	System.err.println
		("strip_DEBUG: Unable to read file " + arguments[0] + '\n'
		+ exception.getMessage ());
	System.exit (-1);
	}

char
	character;
int
	index = 0,
	begin,
	end;
while ((index = Source.indexOf ("DEBUG", index)) >= 0)
	{
	if (index > 0)
		{
		character = Source.charAt (index - 1);
		if (character != '('  &&
			character != ' '  &&
			character != '\t' &&
			character != '\n' &&
			character != '\r')
			{
			index += 5;
			continue;
			}
		}
	begin = Expression_Begin_Line (index);
	end   = Expression_End_Line (index);

	//	Delete the sequence from the Source buffer.
	if ((DEBUG & DEBUG_STRIP) != 0)
		{
		String
			sequence = DEBUG_sequence (begin, end);
		System.err.println
			("Deleting sequence from " + begin + " to " + end + '\n'
			+ sequence);
		}
	Source.delete (begin, end);
	index = begin;
	}

//	Write out the stripped Source buffer.
System.out.print (Source);
System.exit (0);
}


private static String DEBUG_sequence
	(
	int		start,
	int		end
	)
{
return Source.substring (start, end);
}


/**	Find the beginning of the line in which the expression containing
	the specified index starts.
<P>
	Starting with the character immediately preceeding the specified
	index a search is made for the beginning of the line in containing
	the beginning of the expression in which the specified index is
	located. The specified index  is expected to be located within an
	expression such that non-whitespace characters of the expression will
	preceed the index location. The Source is searched backwards from the
	location immediately preceeding the specified index for any
	non-whitespace character. Whitespace characters are ' ', '\t', '\n'
	or '\r'. Then the search continues for the end of the last line. The
	index of the character immediately following the last end of line
	sequence character, or zero if no end of line is found, is returned.
	End of line characters are '\n' or '\r'.
<P>
	@param	index	The character index in the Source before which to
		begin the search.
	@return	The character index in the Source for the beginning of the
		line in which the beginning of the expression containing the
		index location is located.
*/
private static int Expression_Begin_Line
	(
	int		index
	)
{
if ((DEBUG & DEBUG_BEGIN) != 0)
	System.err.println
		("Searching for beginning of expression beginning of line from index "
			+ index);
char
	character;

//	Search for non-whitespace character.
while (--index > 0)
	{
	character = Source.charAt (index);
	if ((DEBUG & DEBUG_BEGIN) != 0)
		System.err.print (character);
	if (character == ' '  ||
		character == '\t' ||
		character == '\n' ||
		character == '\r')
		continue;
	break;
	}

//	Search for the end of the last line.
while (--index > 0)
	{
	character = Source.charAt (index);
	if ((DEBUG & DEBUG_BEGIN) != 0)
		System.err.print (character);
	if (character == '\n' ||
		character == '\r')
		{
		//	Return the beginning of the current line.
		if ((DEBUG & DEBUG_BEGIN) != 0)
			System.err.println
				("Beginning of expression beginning of line: " + (index + 1));
		return ++index;
		}
	}
if ((DEBUG & DEBUG_BEGIN) != 0)
	System.err.println
		("Beginning of expression beginning of line: 0");
return 0;
}

/**	Find the end of the line in which the expression containing
	the specified index ends.
<P>
	Starting with the character at the specified index a search is made
	for the end of the expression containing the specified index. The end
	of the expression is a semicolon character (';'). However, if an open
	brace character ('{') is seen first then the end of the expression is
	the matching close brace character ('}') with nested brace blocks
	skipped (they are part of the expression). During the search for the
	end of the expression characters preceeded by a backslash character
	('\') and single or double quoted strings are skipped.
<P>
	After the end of the expression is located a search is made for the
	next end of line sequence. The location immediately following this
	end of line sequence is returned.
<P>
	@param	index	The character index in the Source after which to
		begin the search.
	@return	The character index in the Source for the end of the
		line in which the end of expression containing the index location is
		located.
*/
private static int Expression_End_Line
	(
	int		index
	)
{
if ((DEBUG & DEBUG_END) != 0)
	System.err.println
		("Searching for end of expression end of line from index " + index);
char
	character;
int
	level = 0;

while (index < Source.length ())
	{
	character = Source.charAt (index++);
	if ((DEBUG & DEBUG_END) != 0)
		System.err.print (character);
	if (character == '\\')
		{
		//	Escaped character.
		if ((DEBUG & DEBUG_END) != 0)
			System.err.print ("[Escape]" + Source.charAt (index));
		index++;
		}
	else if (character == '"' ||
			 character == '\'')
		{
		//	Skip quoted sequence.
		if ((DEBUG & DEBUG_END) != 0)
			System.err.println ("<Quote start]");
		while (index < Source.length () &&
				Source.charAt (index) != character)
			{
			if ((DEBUG & DEBUG_END) != 0)
				System.err.print (Source.charAt (index));
			if (Source.charAt (index++) == '\\')
				{
				if ((DEBUG & DEBUG_END) != 0)
					System.err.print
						("[Escape]" + Source.charAt (index));
				index++;
				}
			}
		if ((DEBUG & DEBUG_END) != 0)
			System.err.println ("[Quote end>" + Source.charAt (index));
		index++;
		}
	else if (character == '{')
		level++;
	else if (character == '}')
		{
		if (--level < 0)
			{
			System.err.println
				("strip_DEBUG: Unbalanced braces in at location " + --index);
			System.exit (2);
			}
		if (level == 0)
			//	End of block.
			break;
		}
	else if (level == 0 && character == ';')
		//	End of statement.
		break;
	}
if ((DEBUG & DEBUG_END) != 0)
	System.err.println
		("\nExpression ends at index " + index);

//	Search for the end of the current line.
while (index < Source.length ())
	{
	character = Source.charAt (index++);
	if (character == '\n' ||
		character == '\r')
		{
		//	Skip the EOL sequence.
		if ((character == '\r' && Source.charAt (index) == '\n') ||
			(character == '\n' && Source.charAt (index) == '\r'))
			++index;
		break;
		}
	}

if ((DEBUG & DEBUG_END) != 0)
	System.err.println
		("End of Expression end of line: " + index);
return index;
}


}	//	End of class
