/*	Pipeline_Source_Manager

PIRL CVS ID: Pipeline_Source_Manager.java,v 1.7 2012/04/16 06:04:10 castalia Exp

Copyright (C) 2008-2012 Arizona Board of Regents on behalf of the Planetary
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package PIRL.Conductor;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.util.Iterator;
import java.util.Vector;
import java.util.regex.Pattern;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableModel;

import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.decorator.ColorHighlighter;
import org.jdesktop.swingx.decorator.Filter;
import org.jdesktop.swingx.decorator.FilterPipeline;
import org.jdesktop.swingx.decorator.PatternFilter;
import org.jdesktop.swingx.decorator.PatternPredicate;

import PIRL.Configuration.Configuration;
import PIRL.Configuration.Configuration_Exception;
import PIRL.Database.Database;
import PIRL.Database.Database_Exception;
import PIRL.Viewers.Parameter_View;

/**	The <i>Pipeline_Source_Manager</i> manages unprocessed source records
	in a Conductor pipeline Sources table.
<p>
	Pipeline_Source_Manager provides a GUI to view any unprocessed source
	records in any Conductor Sources table accessible within the catalog
	specified by the CATALOG parameter in a PVL-style configuration file
	specified at the time the application is launched. The operator can
	optionally delay the processing of selected sources by setting the
	Conductor_ID to the value specified in the static variable {@link
	#DELAY_PROCESSING_STRING}, or release a delayed source record by
	resetting the Conductor_ID to NULL. Source records can also be
	deleted from Sources tables, but will <b>not</b> remove the source
	file specified in the Source_Pathname from the filesystem.
<p>
	<b>Note:</b> The application assumes that all pipeline source tables
	end with the word "Sources."
<p>
	The {@link #Usage() usage} description specifies the application
	command line syntax and describes the GUI elements.
<p>
	@author Rodney Heyd UA/PIRL
	@version 1.7
*/
public class Pipeline_Source_Manager
	extends JFrame
	implements ActionListener 
{
/**	Class ID
*/
public static final String 
	ID = "PIRL.Conductor.Pipeline_Source_Manager (1.7 2012/04/16 06:04:10)";


/** The default configuration file.
*/
public static final String 
	DEFAULT_CONFIGURATION_FILENAME = "Database.conf";
private Configuration The_Configuration;

/**	This string is used to update the Conductor_ID of a source to
	indicate that processing of the source record should be delayed.
<p>
	The "clear delayed source" functionality will only operate on sources
	with a Conductor_ID that matches this value.
*/
public static final String 
	DELAY_PROCESSING_STRING = "Source Delayed";

private Database The_Database;
private String Catalog;
private Vector<String> Source_Tables;
private Vector<Vector> Sources_List;
private Vector<String> Column_Names;
private int Source_Pathname_Column;

// GUI Elements

private JSplitPane Split_Pane;
private JComboBox  Source_Tables_ComboBox;
private JXTable Sources_Display;
private Pipeline_Manager_Table_Model Sources_Table_Model;
private JTextField Source_Pathname_Filter;
private JButton 
	Delay_Selected_Sources,
	Delete_Selected_Sources,
	Process_Selected_Sources,
	Refresh_Sources_List;
private JMenuItem 
	Select_All,
	Select_Delayed,
	Select_Nulls,
	Select_None;

/** Exit Status Values
 */
public static final int
	EXIT_SUCCESS					= 0,
	EXIT_DATABASE_ERROR				= 1,
	EXIT_CONFIGURATION_ERROR 		= 2,
	EXIT_COMMAND_LINE_SYNTAX_ERROR 	= 3;

private static String
	NL								= System.getProperty ("line.separator");


// Debug control.
private static final int
	DEBUG_OFF		= 0,
	DEBUG_MAIN 		= 1 << 0,
	DEBUG_INIT		= 1 << 2,
	DEBUG_EVENTS	= 1 << 3,
	DEBUG			= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/** Construct a Pipeline_Source_Manager using a Configuration.
<p>
	A database will be established and the GUI will be displayed.
<p>
	@param configuration
*/
public Pipeline_Source_Manager(Configuration configuration)
{
	super("Pipeline_Source_Manager");
	The_Configuration = configuration;
	init();
	Create_GUI();
}


private Pipeline_Source_Manager()
{}

/*==============================================================================
	Helpers
*/
/** Initializes the class. Initiates the database connection, gets the list
 of sources tables, etc.
 */
private void init()
{

	Catalog = The_Configuration.Get_One("CATALOG");
	try 
	{
		The_Database = new Database(The_Configuration);
		The_Database.Connect();

		//	Mask out all password parameters.
		The_Configuration.Case_Sensitive (false);
		The_Configuration.Set_All ("PASSWORD", "****");
	} 
	catch (Database_Exception e) 
	{
		System.err.println(
				"A database error occurred while attempting to establish a " +
			    "connection to the database:" + NL +
				NL +
				e.getMessage());
		System.exit(EXIT_DATABASE_ERROR);
	} 
	catch (Configuration_Exception e) 
	{
		System.err.println(
			    "Unable to get the database connection configuration parameters:" + NL +
				NL +
			    e.getMessage()
				);
		System.exit(EXIT_CONFIGURATION_ERROR);
	}
	
	Source_Tables = Get_Source_Tables();
	
	if(Source_Tables.size() == 0)
	{
		String warning_message = "No pipeline source tables were found in the " + 
		Catalog + " catalog.  Either no source tables exist, or the account " +
				"you are using lacks sufficient privileges on the catalog.";
		PIRL.Viewers.Dialog_Box.Warning(warning_message, this);
		System.exit(EXIT_DATABASE_ERROR);
	}

	Get_Sources(Source_Tables.firstElement());
	Sources_Table_Model = new Pipeline_Manager_Table_Model();
	Sources_Table_Model.setDataVector(Sources_List, Column_Names);
	if((DEBUG & DEBUG_INIT) != 0)
	{
		for(String source_table: Source_Tables)
			System.out.println("Found Source Table==>" + source_table);
		System.out.println("Column_Names==>" + Column_Names);
	}

}


private class Pipeline_Manager_Table_Model
	extends DefaultTableModel
{
	public boolean isCellEditable(int row, int column)
	{
		// No cells should be editable.
		return false;
	}

}

/** This method gets the list of pipeline sources tables from the database.
@return A vector of strings that are the names of the sources tables contained
within the catalog specified within the configuration file. This is list is
displayed in the drop box in the GUI.
 */
private Vector<String> Get_Source_Tables()
{
	Vector<String> source_tables = null;
	try 
	{
		source_tables = The_Database.Tables(Catalog);
	} 
	catch (Database_Exception e) 
	{
		PIRL.Viewers.Dialog_Box.Error( 
			    "A database error occurred while retrieving the list of pipeline " +
			    "sources tables in the " + Catalog + " catalog." + NL +
				NL +
			    "The list of source tables may be incomplete. Restarting the " +
			    "application is recommended:" + NL + 
				NL +
				e.getMessage()
				);
	}

	Iterator table_iterator = source_tables.iterator();
	while (table_iterator.hasNext())
	{
		String table_name = table_iterator.next().toString();
		if(!table_name.endsWith("Sources"))
			table_iterator.remove();
	}
	return source_tables;
	
}

/** Get the list of pending and delayed sources in the specified table.
@param table - A string containing the name of the table from which a list
of sources is to be retrieved.
 */
private void Get_Sources(String table)
{
	String qualified_table = Catalog + "." + table;
	String conditional = 
		"Conductor_ID is NULL OR Conductor_ID = '" + 
		DELAY_PROCESSING_STRING + "'";
	
	try 
	{
		Sources_List = The_Database.Select(qualified_table, conditional);
	} 
	catch (Database_Exception e) 
	{
		PIRL.Viewers.Dialog_Box.Error( 
			    "A database error occurred while retrieving the list of available " +
			    "sources from the database:" + NL + 
				NL +
				e.getMessage() 
				);
	}
	
	Column_Names = Sources_List.get(0);
	Sources_List.remove(Column_Names);
	return;
}

private void Delay_Sources()
{
	int [] selected_rows = Sources_Display.getSelectedRows();
	
	int source_number_index = Column_Names.indexOf("Source_Number");
	Vector<String> Column_To_Update = new Vector<String>();
	Column_To_Update.add("Conductor_ID");
	Vector<String> Values = new Vector<String>();
	Values.add(DELAY_PROCESSING_STRING);
	String table = 
		Source_Tables_ComboBox.getSelectedItem().toString();
	String warning_message = "The following source numbers were not " +
			"updated:" + NL;
	boolean update_successfull = false;
	boolean failed_updates = false;
	for(int row: selected_rows)
	{
		int model_row_index = Sources_Display.convertRowIndexToModel(row);
		int source_number = Integer.parseInt(
			Sources_Table_Model.getValueAt(
					model_row_index, 
					source_number_index).toString());		
		try 
		{
			update_successfull = Delay_Source(source_number, table);
		} 
		catch (Database_Exception e) 
		{
			PIRL.Viewers.Dialog_Box.Error( 
			    "A database error occurred while attempting to set the " +
			    "Conductor ID to " + DELAY_PROCESSING_STRING + " for source " +
			    "number " + source_number + ":" + NL + 
				NL +
				e.getMessage() );
		}
		if(!update_successfull)
		{
			failed_updates = true;
			warning_message +=source_number + NL;
		}
		
	}
	
	if(failed_updates)
	{
		warning_message += NL +
			"A conductor process may have begun processing" + NL +
			"these sources." + NL;
		PIRL.Viewers.Dialog_Box.Warning(warning_message, this);
	}
}


private void Process_Sources()
{
	int [] selected_rows = Sources_Display.getSelectedRows();
	
	String warning_message = "The following sources were not reset:" + NL;
	int source_number_index = Column_Names.indexOf("Source_Number");
	boolean source_updated;
	boolean updates_failed = false;
	for(int row: selected_rows)
	{
		source_updated = false;
		int model_row_index = Sources_Display.convertRowIndexToModel(row);
		int source_number = Integer.parseInt(
			Sources_Table_Model.getValueAt(
					model_row_index, 
					source_number_index).toString());		
		try 
		{
			source_updated = 
				Process_Source( source_number, Source_Tables_ComboBox.getSelectedItem().toString());
		} 
		catch (Database_Exception e) 
		{
			if(DEBUG != 0)
				e.printStackTrace();
			
			PIRL.Viewers.Dialog_Box.Error( 
			    "A database error occurred while attempting to set the " +
			    "Conductor ID to NULL for source number " + source_number + 
			    ":" + NL +
				NL +
				e.getMessage());
		}
		
		if(!source_updated)
		{
			warning_message += source_number + NL;
			updates_failed = true;
		}
		
	}
	
	warning_message += NL +
		"This is probably the result of selecting a source " +
		"that has already been reset." + NL;

	/* Note: Uncomment the code below to display a dialog when sources are 
	 * selected that were not reset.  This dialog is rather superfluous as
	 * selecting a source that has not been delayed doesn't actually change
	 * anything.  But if you want to know when this happens, uncomment the
	 * code below.
	 */
/*	if(updates_failed)
	{
		PIRL.Viewers.Dialog_Box.Warning( warning_message );
	}
*/
}


private void Delete_Sources()
{
	int [] selected_rows = Sources_Display.getSelectedRows();
	String warning_message = "The following source numbers were not " +
	"deleted:" + NL;
	
	int source_number_index = Column_Names.indexOf("Source_Number");
	boolean delete_successfull = false;
	boolean failed_deletions = false;
	for(int row: selected_rows)
	{
		int model_row_index = Sources_Display.convertRowIndexToModel(row);
		int source_number = Integer.parseInt(
			Sources_Table_Model.getValueAt(
					model_row_index, 
					source_number_index).toString());
		try
		{
			delete_successfull = 
				Delete_Source(source_number, 
				              Source_Tables_ComboBox.getSelectedItem().toString());
		}
		catch ( Database_Exception e )
		{
			if(DEBUG != 0)
				e.printStackTrace();
			PIRL.Viewers.Dialog_Box.Error( 
			    "A database error occurred while attempting to delete source " +
			    "number " + source_number + ":" + NL +
				NL +
				e.getMessage());
		}
		
		if(!delete_successfull)
		{
			warning_message += source_number + NL;
			failed_deletions = true;
		}
	}
	if(failed_deletions)
	{
		warning_message += NL +
			"A conductor process may have begun processing" + NL +
			"these sources." + NL;
		PIRL.Viewers.Dialog_Box.Warning(warning_message, this);
	}
}

/*=============================================================================
	GUI
*/
private void Create_GUI()
{
	setSize(1024, 768);
	setJMenuBar(Menu_Bar());
	Split_Pane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
	
	/*========================================================================
	 Top panel components
	 */
	JPanel top_panel = new JPanel(new BorderLayout());
	top_panel.setSize(1024, 400);
	// The combobox table selector
	JPanel table_panel = new JPanel();
	JLabel source_table_label = new JLabel("Source Table:");
	Source_Tables_ComboBox = new JComboBox(Source_Tables);
	Source_Tables_ComboBox.addActionListener(this);
	source_table_label.setLabelFor(Source_Tables_ComboBox);
	Split_Pane.setTopComponent(top_panel);
	table_panel.add(source_table_label);
	table_panel.add(Source_Tables_ComboBox);
	top_panel.add(table_panel, BorderLayout.NORTH);
	
	
	// Source_Pathname filter
	JPanel filter_panel = new JPanel();
	JLabel pathname_filter_label = new JLabel("Source Pathname Filter:");
	Source_Pathname_Filter = new JTextField();
	Source_Pathname_Filter.setColumns(50);
	pathname_filter_label.setLabelFor(Source_Pathname_Filter);
	filter_panel.add(pathname_filter_label);
	filter_panel.add(Source_Pathname_Filter);
	top_panel.add(filter_panel, BorderLayout.CENTER);
	Source_Pathname_Column = Sources_Table_Model.findColumn("Source_Pathname");
	Source_Pathname_Filter.getDocument().addDocumentListener(
			new DocumentListener() 
			{

				public void changedUpdate(DocumentEvent arg0) 
				{
				}

				public void insertUpdate(DocumentEvent arg0) {
					String filter_text = Source_Pathname_Filter.getText ();
					Sources_Display.setFilters ( new FilterPipeline(
							new Filter[] {
									new PatternFilter(
											filter_text + ".*", 
											0, 
											Source_Pathname_Column)}) );
				
				}

				public void removeUpdate(DocumentEvent arg0) {
					String filter_text = Source_Pathname_Filter.getText ();
					Sources_Display.setFilters ( new FilterPipeline(
							new Filter[] {
									new PatternFilter(
											filter_text + ".*", 
											0, 
											Source_Pathname_Column)}) );
				
				}
		
			});
	
	// Delay Selected Sources
	
	Delay_Selected_Sources = new JButton("Delay Selected Sources");
	Delay_Selected_Sources.addActionListener(this);
	Delay_Selected_Sources.setEnabled(false);
	
	// Process Selected Sources
	
	Process_Selected_Sources = new JButton("Clear Delay Flag On Selected Sources");
	Process_Selected_Sources.addActionListener(this);
	Process_Selected_Sources.setEnabled(false);

	// Delete Selected Sources
	
	Delete_Selected_Sources = new JButton("Delete Selected Sources");
	Delete_Selected_Sources.addActionListener(this);
	Delete_Selected_Sources.setEnabled(false);
	
	// Refresh Sources List
	
	Refresh_Sources_List = new JButton("Refresh Sources List");
	Refresh_Sources_List.addActionListener( this );
	Refresh_Sources_List.setEnabled( true );
	
	JPanel button_panel = new JPanel();
	button_panel.add(Delay_Selected_Sources);
	button_panel.add(Process_Selected_Sources);
	button_panel.add(Delete_Selected_Sources);
	button_panel.add( Refresh_Sources_List );
	
	top_panel.add(button_panel, BorderLayout.SOUTH);
	/*========================================================================
	 Bottom panel components (the table)
	 */
	Sources_Display = new JXTable(Sources_Table_Model);
	Sources_Display.setColumnControlVisible(true);
	Sources_Display.packAll();
	Sources_Display.setHorizontalScrollEnabled(true);
	
	Sources_Display.getSelectionModel().addListSelectionListener(
			new ListSelectionListener() {

				public void valueChanged(ListSelectionEvent arg0) {
					int[] selected_rows = Sources_Display.getSelectedRows();
					if(selected_rows.length > 0)
					{
						Delay_Selected_Sources.setEnabled( true );
						Process_Selected_Sources.setEnabled( true );
						Delete_Selected_Sources.setEnabled( true );
					}
					else
					{
						Delay_Selected_Sources.setEnabled( false );
						Process_Selected_Sources.setEnabled( false );
						Delete_Selected_Sources.setEnabled( false );
					}
				}
		
	});
	set_table_highlighters();
	JScrollPane scrollpane = new JScrollPane(Sources_Display);
	scrollpane.setSize(1024, 500);
	Split_Pane.setBottomComponent(scrollpane);
	
	add(Split_Pane);
}

private JMenuBar Menu_Bar()
{
	JMenuBar MenuBar = new JMenuBar();
	
	JMenu file_menu = new JMenu("File");
	MenuBar.add(file_menu);
	
	JMenuItem view_configuration = new JMenuItem("View Configuration");
	
	// Configuration Viewer.  Usefull for viewing or debuging the
	// configuration.
	view_configuration.addActionListener (
		new ActionListener() {

			public void actionPerformed (
					ActionEvent e )
			{
				Parameter_View config_view = 
					new Parameter_View( The_Configuration);
				config_view.setVisible ( true );
			}
			
		});
	
	file_menu.add(view_configuration);
	
	JMenuItem exit_app = new JMenuItem("Exit", KeyEvent.VK_X);
	exit_app.setEnabled( true );
	exit_app.setActionCommand( "Exit" );
	exit_app.addActionListener ( 
			new ActionListener(){

				public void actionPerformed ( ActionEvent e )
				{

					System.exit ( EXIT_SUCCESS );
					
				}
				
			});
	
	file_menu.add(exit_app);
	
	JMenu select_menu = new JMenu("Select");
	Select_All = new JMenuItem("Select All");
	Select_All.addActionListener(this);
	
	select_menu.add(Select_All);
	Select_Delayed = new JMenuItem("Select Delayed");
	Select_Delayed.addActionListener(this);
	select_menu.add(Select_Delayed);
	Select_Nulls = new JMenuItem("Select Nulls");
	select_menu.add(Select_Nulls);
	Select_Nulls.addActionListener(this);
	Select_None = new JMenuItem("Deselect All");
	select_menu.add(Select_None);
	Select_None.addActionListener(this);
	
	MenuBar.add(select_menu);
	

	return MenuBar;
}

private void set_table_highlighters()
{
	int conductor_id_column = 
		Sources_Table_Model.findColumn ( "Conductor_ID" );
	Pattern delayed_pattern = Pattern.compile( DELAY_PROCESSING_STRING );
	PatternPredicate color_predicate = 
		new PatternPredicate(
		       delayed_pattern,
		       conductor_id_column);
	ColorHighlighter delayed_highlighter = new ColorHighlighter();
	delayed_highlighter.setHighlightPredicate( color_predicate );
	delayed_highlighter.setForeground( Color.BLUE);
	delayed_highlighter.setBackground( Color.WHITE );
	delayed_highlighter.setSelectedBackground( Color.BLUE );
	delayed_highlighter.setSelectedForeground( Color.WHITE );
	Sources_Display.addHighlighter( delayed_highlighter );
	
}

/*==============================================================================
Source methods
 */

/** This method deletes a source number from a specified table.  

@param source_number - the source_number of the source to be deleted.
@param table - the table the source is to be deleted from
@return true if the sources was successfully deleted, false otherwise. A false
return value means that either the source id doesn't exist, or a conductor
has started processing the source.
@throws Database_Exception if a database error occurs while attempting to
delete the source.
*/
public boolean Delete_Source(int source_number, String table) throws Database_Exception
{
	boolean source_deleted = true;
	String warning_message = "The following source numbers were not " +
	"deleted:" + NL;
	String qualified_table = 
		Catalog + "." + table;
	String condition_clause;
	int deleted_row_count = 0;
	condition_clause = 
		"Source_Number=" + source_number + " AND " +
				"(Conductor_ID is NULL OR Conductor_ID = '" +
				DELAY_PROCESSING_STRING + "')";
	deleted_row_count = The_Database.Delete(qualified_table, condition_clause);

	if(deleted_row_count == 0)
	{
		source_deleted = false;
		warning_message +=source_number + NL;
	}

	return source_deleted;
}

/** This method makes a source available for processing that was delayed using
the {@link #Delay_Source(int, String)} method.  It will only reset sources for which the
Conductor ID field has been set to the value of {@value #DELAY_PROCESSING_STRING}

@param source_number - the source_number to operate on
@param table - the table containing the sources to be reset
@return true if the database update was successfull.
@throws Database_Exception if a database error occurred during the update.
@see #Delay_Source(int, String)
@see #DELAY_PROCESSING_STRING
 */
public boolean Process_Source(int source_number, String table) throws Database_Exception
{

	Vector<String> Column_To_Update = new Vector<String>();
	Column_To_Update.add("Conductor_ID");
	Vector<String> Values = new Vector<String>();
	Values.add("NULL");
	String qualified_table = 
		Catalog + "." + table;
	String condition_clause;
	condition_clause = 
		"Source_Number=" + source_number + " AND Conductor_ID='" +
		DELAY_PROCESSING_STRING + "'";

	int records_updated = 
			The_Database.Update(qualified_table, Column_To_Update, Values, condition_clause);

	if( records_updated == 1 )
		return true;
	else
		return false;

}

/*==============================================================================
GUI Event Handlers
*/
public void actionPerformed(ActionEvent event) 
{
	Object source = event.getSource();
	if((DEBUG & DEBUG_EVENTS) != 0)
	{
		System.out.println("Caught event from " + source.getClass().getName());
	}
	
	if(source.equals(Source_Tables_ComboBox))
	{
		Get_Sources(Source_Tables_ComboBox.getSelectedItem().toString());
		Sources_Table_Model.setDataVector(Sources_List, Column_Names);
		Sources_Table_Model.fireTableDataChanged();
		Sources_Display.packAll();
	}
	
	if(source.equals(Delay_Selected_Sources))
	{
		Delay_Sources();
		Get_Sources(Source_Tables_ComboBox.getSelectedItem().toString());
		Sources_Table_Model.setDataVector(Sources_List, Column_Names);
		Sources_Table_Model.fireTableDataChanged();
		Sources_Display.packAll();
	}
	
	if(source.equals(Process_Selected_Sources))
	{
		Process_Sources();
		Get_Sources(Source_Tables_ComboBox.getSelectedItem().toString());
		Sources_Table_Model.setDataVector(Sources_List, Column_Names);
		Sources_Table_Model.fireTableDataChanged();
		Sources_Display.packAll();
	}
	
	if(source.equals(Delete_Selected_Sources))
	{
		Delete_Sources();
		Get_Sources(Source_Tables_ComboBox.getSelectedItem().toString());
		Sources_Table_Model.setDataVector(Sources_List, Column_Names);
		Sources_Table_Model.fireTableDataChanged();
		Sources_Display.packAll();
	}
	
	if(source.equals( Refresh_Sources_List ))
	{
		Get_Sources(Source_Tables_ComboBox.getSelectedItem().toString());
		Sources_Table_Model.setDataVector( Sources_List, Column_Names );
		Sources_Table_Model.fireTableDataChanged();
		Sources_Display.packAll();
	}
	
	if(source.equals(Select_All))
	{
		Sources_Display.selectAll();
	}
	
	if(source.equals(Select_Delayed))
	{
		Sources_Display.clearSelection();
		int conductor_index = Column_Names.indexOf("Conductor_ID");
		int row_count = Sources_Table_Model.getRowCount();
		
		for(int row = 0; row < row_count; row++)
		{
			Object cell_value = Sources_Table_Model.getValueAt(row, conductor_index);
			
			if(cell_value != null && cell_value
					.equals(DELAY_PROCESSING_STRING))
			{
				Sources_Display.addRowSelectionInterval(row, row);
			}
		}
	}
	
	if(source.equals(Select_Nulls))
	{
		Sources_Display.clearSelection();
		int conductor_index = Column_Names.indexOf("Conductor_ID");
		int row_count = Sources_Table_Model.getRowCount();
		
		for(int row = 0; row < row_count; row++)
		{
			if(Sources_Table_Model.getValueAt(row, conductor_index) == null)
			{
				Sources_Display.addRowSelectionInterval(row, row);
			}
		}
	}
	
	if(source.equals(Select_None))
		Sources_Display.clearSelection();
}


/** This method will set the Conductor ID field of a source record to the
{@link #DELAY_PROCESSING_STRING} to cause a conductor to skip the specified
source.  This gives the operator a very limited ability to prioritize sources by
delaying the processing of some source fields while allowing others to be
processed.

@param source_number - the source number to be delayed
@param table - the pipeline source table containing the source to be delayed
@return true if the Conductor ID of the given source was successfully updated.
A false return value indicates that either the source does not exist, or a
conductor has already begun processing the source.
@throws Database_Exception if a database error occurs while attempting to update
the source record.
@see #Process_Source(int, String)
@see #DELAY_PROCESSING_STRING
 */
public boolean Delay_Source(int source_number, String table) throws Database_Exception
{
	Vector<String> Column_To_Update = new Vector<String>();
	Column_To_Update.add("Conductor_ID");
	Vector<String> Values = new Vector<String>();
	Values.add(DELAY_PROCESSING_STRING);
	String qualified_table = 
		Catalog + "." + table;
	String condition_clause;
	int updated_row_count;
	condition_clause = 
		"Source_Number=" + source_number + " AND Conductor_ID is NULL";
	updated_row_count = 
		The_Database.Update(
				qualified_table, 
				Column_To_Update, 
				Values, 
				condition_clause);
	if(updated_row_count == 0)
		return false;
	else
		return true;
}

/*==============================================================================
	Application Methods
*/
/**	Manage sources in a Conductor Pipeline Source Table.
<p>
	The command line syntax is described in the {@link #Usage()} method.
<p>
Exit Status Values:
<p><dl>
<dt>{@link #EXIT_SUCCESS}
<dd>No errors occurred.

<dt>{@link #EXIT_DATABASE_ERROR}
<dd>There was a problem accessing the database.

<dt>{@link #EXIT_CONFIGURATION_ERROR}
<dd>There was a problem with the Configuration file.

<dt>{@link #EXIT_COMMAND_LINE_SYNTAX_ERROR}
<dd>The command line contains invalid syntax.
</dl><p>
	@param arguments	The {@link #Usage() command line arguments}.
*/
public static void main(String[] arguments) 
{
	String configuration_filename = null;
	for (int count = 0;
	 count < arguments.length;
	 count++)
	{
	if (arguments[count].length () == 0)
	continue;
	if (arguments[count].charAt (0) == '-')
	{
	switch (arguments[count].charAt (1))
		{
		case 'C':   // -Database Configuration
		case 'c':
			String name = null;
			if (++count == arguments.length ||
					arguments[count].charAt (0) == '-')
					{
					//	Use the default configuration file.
					name = DEFAULT_CONFIGURATION_FILENAME;
					--count;
					}
				else
					name = arguments[count];
				if (configuration_filename == null)
					configuration_filename = name;
				else
					{
					Usage ();
					}
				break;
		case 'H':	//	-Help
		case 'h':
			Usage ();
			System.exit ( EXIT_SUCCESS );
		default:
			System.out.println
					("Unrecognized switch: " + arguments[count]);
			Usage ();
			System.exit(EXIT_COMMAND_LINE_SYNTAX_ERROR);
		}
	}
	}
	if(configuration_filename == null)
		configuration_filename=DEFAULT_CONFIGURATION_FILENAME;
		
	Configuration configuration=null;
	try 
	{
		configuration = new Configuration(configuration_filename);
	} 
	catch (Configuration_Exception e) 
	{
		System.err.println(
			    "A configuration error occured while reading the configuration " +
			    "file: " + configuration_filename + NL +
				NL +
				e.getMessage() 
				);
		System.exit(EXIT_CONFIGURATION_ERROR);
	}
	
	final Configuration Configuration = configuration;
	SwingUtilities.invokeLater(
			new Runnable() {

				public void run() {
					Pipeline_Source_Manager manager = 
						new Pipeline_Source_Manager(Configuration);
					manager.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
					manager.setVisible(true);
					
				}
				
			});
		

}

/**	Prints the command line usage syntax.
<p>
<blockquote><pre>
Usage: <b>Pipeline_Source_Manager</b> &lt;<i>Options</i>&gt;
&nbsp;&nbsp;Options -
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>C</u>onfiguration</b> &lt;<i>source</i>&gt;
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Default: Database.conf
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>H</u>elp</b>]
</pre></blockquote>
<p>
<h3>Configuration
</h3><p>
	Pipeline_Source_Manager requires a Configuration file which
	contains database and catalog connection information. The provider
	Conductor.conf file is a suitable template to use for this purpose.
<p>
<h3>
	The GUI consists of:
</h3>
<dl>
<dt>A Source Table selector
<dd>Used to select the effective Sources table.

<dt>A Source_Pathname filter
<dd>A full or partial pathname is used to filter the source records list
	to contain only the records with matching Source_Pathname field
	values.

<dt>Operations buttons
<dd>Used to perform the coresponding delay/undelay or delete operations
	on the source records selected in the Sources table.
<p>
	<b>Note:</b> If a Conductor has started processing a source record after
	the time that a Sources table was read from the database, the
	Pipeline_Source_Manager will not operate on that source record.  This
	prevents the Conductor_ID from being set to a delayed status if a
	Conductor has already acquired the source record.

<dt>Sources table
<dd>A table listing the unprocessed and delayed sources found in the
	currently selected Sources Table. One or more of the source records
	listed in the table may be selected and then an Operation button used
	to perform the corresponding operation on the selected source
	records.
</dl>
*/
public static void Usage() 
{
	System.out.println
	(
	ID + NL +
	"Usage: Pipeline_Source_Manager <Options>" + NL +
	"  Options -" + NL +
	"    [-Configuration <source>]" + NL +
	"      Default: " + DEFAULT_CONFIGURATION_FILENAME + NL +
	"    [-Help]"
	);	
}
}
